import { Stats } from '@/lib/types'

interface AdminStatsProps {
  stats: {
    totalMarkets: number
    activeMarkets: number
    totalUsers: number
    totalVolume: number
    totalBets: number
  }
}

export default function AdminStats({ stats }: AdminStatsProps) {
  const formatVolume = (volume: number) => {
    if (volume >= 1000000) {
      return `$${(volume / 1000000).toFixed(1)}M`
    } else if (volume >= 1000) {
      return `$${(volume / 1000).toFixed(1)}K`
    } else {
      return `$${volume.toFixed(0)}`
    }
  }

  const statCards = [
    {
      title: 'Total Markets',
      value: stats.totalMarkets,
      icon: '📈',
      color: 'blue'
    },
    {
      title: 'Active Markets',
      value: stats.activeMarkets,
      icon: '🟢',
      color: 'green'
    },
    {
      title: 'Total Users',
      value: stats.totalUsers,
      icon: '👥',
      color: 'purple'
    },
    {
      title: 'Total Volume',
      value: formatVolume(stats.totalVolume),
      icon: '💰',
      color: 'yellow'
    },
    {
      title: 'Total Bets',
      value: stats.totalBets,
      icon: '🎯',
      color: 'red'
    }
  ]

  const getColorClasses = (color: string) => {
    switch (color) {
      case 'blue':
        return 'bg-blue-500 text-white'
      case 'green':
        return 'bg-green-500 text-white'
      case 'purple':
        return 'bg-purple-500 text-white'
      case 'yellow':
        return 'bg-yellow-500 text-white'
      case 'red':
        return 'bg-red-500 text-white'
      default:
        return 'bg-gray-500 text-white'
    }
  }

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6">
      {statCards.map((stat, index) => (
        <div key={index} className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center">
            <div className={`w-12 h-12 rounded-lg ${getColorClasses(stat.color)} flex items-center justify-center text-xl`}>
              {stat.icon}
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-black">{stat.title}</p>
              <p className="text-2xl font-bold text-black">{stat.value}</p>
            </div>
          </div>
        </div>
      ))}
    </div>
  )
}
