'use client'

import React, { memo } from 'react'
import Link from 'next/link'

interface TransactionWithDetails {
  id: string
  user_id: string
  type: string
  amount: number
  currency: string
  status: string
  description: string
  created_at: string
  processed_at: string | null
  external_transaction_id: string | null
  payment_method: any
  market_id: string | null
  user_email: string
  market_title?: string
  fee_amount: number
  net_amount: number
}

interface AdminTransactionsTableProps {
  transactions: TransactionWithDetails[]
  loading: boolean
  hasMore: boolean
  onLoadMore: () => void
  onRefresh: () => void
}

const AdminTransactionsTable = memo(({ transactions, loading, hasMore, onLoadMore, onRefresh }: AdminTransactionsTableProps) => {
  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    })
  }

  const getTypeColor = (type: string) => {
    switch (type) {
      case 'deposit':
        return 'text-green-600 bg-green-100'
      case 'withdrawal':
        return 'text-red-600 bg-red-100'
      case 'bet':
        return 'text-blue-600 bg-blue-100'
      case 'win':
        return 'text-yellow-600 bg-yellow-100'
      case 'fee':
        return 'text-purple-600 bg-purple-100'
      default:
        return 'text-gray-600 bg-gray-100'
    }
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
        return 'text-green-600 bg-green-100'
      case 'pending':
        return 'text-yellow-600 bg-yellow-100'
      case 'processing':
        return 'text-blue-600 bg-blue-100'
      case 'failed':
        return 'text-red-600 bg-red-100'
      case 'cancelled':
        return 'text-gray-600 bg-gray-100'
      default:
        return 'text-gray-600 bg-gray-100'
    }
  }

  return (
    <div className="bg-white rounded-lg shadow-sm overflow-hidden">
      <div className="overflow-x-auto">
        <table className="min-w-full divide-y divide-gray-200">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                User & Details
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Type & Status
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Amount
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Payment Method
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Date
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Actions
              </th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {transactions.map((transaction) => (
              <tr key={transaction.id} className="hover:bg-gray-50">
                <td className="px-6 py-4">
                  <div>
                    <div className="text-sm font-medium text-gray-900">
                      {transaction.user_email}
                    </div>
                    <div className="text-sm text-gray-500 line-clamp-2">
                      {transaction.description || 'No description'}
                    </div>
                    <div className="text-xs text-gray-400">
                      ID: {transaction.id.slice(0, 8)}...
                    </div>
                    {transaction.market_id && (
                      <div className="text-xs text-blue-600">
                        Market: {transaction.market_id.slice(0, 8)}...
                      </div>
                    )}
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div className="space-y-2">
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getTypeColor(transaction.type)}`}>
                      {transaction.type}
                    </span>
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(transaction.status)}`}>
                      {transaction.status}
                    </span>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div>
                    <div className={`text-sm font-bold ${
                      transaction.amount > 0 ? 'text-green-600' : 'text-red-600'
                    }`}>
                      {transaction.amount > 0 ? '+' : ''}{formatCurrency(transaction.amount)}
                    </div>
                    {transaction.fee_amount && transaction.fee_amount > 0 && (
                      <div className="text-xs text-gray-500">
                        Fee: {formatCurrency(transaction.fee_amount)}
                      </div>
                    )}
                    {transaction.net_amount !== transaction.amount && (
                      <div className="text-xs text-gray-500">
                        Net: {formatCurrency(transaction.net_amount)}
                      </div>
                    )}
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div>
                    <div className="text-sm text-gray-900">
                      {transaction.payment_method?.type || 'Unknown'}
                    </div>
                    {transaction.payment_method?.last_four && (
                      <div className="text-xs text-gray-500">
                        ****{transaction.payment_method.last_four}
                      </div>
                    )}
                    <div className="text-xs text-gray-500">
                      {transaction.currency.toUpperCase()}
                    </div>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div className="text-sm text-gray-900">
                    {formatDate(transaction.created_at)}
                  </div>
                  {transaction.processed_at && (
                    <div className="text-xs text-gray-500">
                      Processed: {formatDate(transaction.processed_at)}
                    </div>
                  )}
                </td>
                <td className="px-6 py-4">
                  <div className="flex space-x-2">
                    {transaction.market_id && (
                      <Link
                        href={`/markets/${transaction.market_id}`}
                        className="text-blue-600 hover:text-blue-900 text-sm"
                      >
                        View Market
                      </Link>
                    )}
                    {transaction.external_transaction_id && (
                      <a
                        href="#"
                        className="text-gray-600 hover:text-gray-900 text-sm"
                      >
                        External: {transaction.external_transaction_id.slice(0, 8)}...
                      </a>
                    )}
                  </div>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>

      {transactions.length === 0 && !loading && (
        <div className="text-center py-12">
          <div className="text-gray-500 mb-2">No transactions found</div>
          <p className="text-sm text-gray-400">
            No transactions match the current filters
          </p>
        </div>
      )}

      {hasMore && (
        <div className="px-6 py-4 border-t border-gray-200 bg-gray-50">
          <button
            onClick={onLoadMore}
            disabled={loading}
            className="w-full bg-blue-600 text-white py-2 px-4 rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {loading ? 'Loading...' : 'Load More Transactions'}
          </button>
        </div>
      )}
    </div>
  )
})

AdminTransactionsTable.displayName = 'AdminTransactionsTable'

export default AdminTransactionsTable
