'use client'

import React, { memo, useState } from 'react'
import { AdminUser } from '@/lib/hooks/useAdminUsersOptimized'
import { LoadingSpinner, ButtonLoadingSpinner } from '@/components/ui/LoadingSpinner'
import { useToast } from '@/components/ui/Toast'
import { supabase } from '@/lib/supabase-client'

interface AdminUsersTableProps {
  users: AdminUser[]
  loading: boolean
  hasMore: boolean
  onLoadMore: () => void
  onRefresh: () => void
}

const UserRow = memo(({ user, onAction }: { 
  user: AdminUser, 
  onAction: (userId: string, action: string) => Promise<void> 
}) => {
  const [actionLoading, setActionLoading] = useState(false)

  const handleAction = async (action: string) => {
    setActionLoading(true)
    try {
      await onAction(user.id, action)
    } finally {
      setActionLoading(false)
    }
  }

  const getRiskColor = (riskScore: number) => {
    if (riskScore < 30) return 'text-green-600 bg-green-100'
    if (riskScore < 60) return 'text-yellow-600 bg-yellow-100'
    return 'text-red-600 bg-red-100'
  }

  const getStatusBadges = (user: AdminUser) => {
    const badges = []
    
    if (user.is_admin) badges.push({ text: 'Admin', color: 'bg-purple-100 text-purple-800' })
    if (user.is_banned) badges.push({ text: 'Banned', color: 'bg-red-100 text-red-800' })
    if (user.kyc_status === 'approved') badges.push({ text: 'KYC Verified', color: 'bg-green-100 text-green-800' })
    else if (user.kyc_status === 'pending') badges.push({ text: 'KYC Pending', color: 'bg-yellow-100 text-yellow-800' })
    else if (user.kyc_status === 'rejected') badges.push({ text: 'KYC Rejected', color: 'bg-red-100 text-red-800' })
    
    return badges
  }

  return (
    <tr className="hover:bg-gray-50">
      <td className="px-6 py-4">
        <div className="flex items-center">
          <div className="w-8 h-8 bg-gradient-to-br from-pink-500 via-purple-500 to-white rounded-full flex items-center justify-center mr-3">
            <span className="text-white text-sm font-bold">
              {user.username?.charAt(0).toUpperCase() || 'U'}
            </span>
          </div>
          <div>
            <div className="text-sm font-medium text-black">{user.username || 'No username'}</div>
            <div className="text-sm text-black">{user.email}</div>
          </div>
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap">
        <div className="space-y-1">
          {getStatusBadges(user).map((badge, index) => (
            <span key={index} className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${badge.color}`}>
              {badge.text}
            </span>
          ))}
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
        <div className="space-y-1">
          <div className="text-black">Win Rate: {user.win_rate.toFixed(1)}%</div>
          <div className="text-black">Bets: {user.bet_count}</div>
          <div className="text-black">Profit: ${user.total_profit.toFixed(2)}</div>
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap">
        <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getRiskColor(user.risk_score)}`}>
          {user.risk_score}/100
        </span>
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
        <div className="space-y-1">
          <div className="text-black">Volume: ${user.total_volume.toLocaleString()}</div>
          <div className="text-black">Last: {new Date(user.last_activity).toLocaleDateString()}</div>
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
        <div className="flex space-x-2">
          {!user.is_banned ? (
            <button
              onClick={() => handleAction('ban')}
              disabled={actionLoading}
              className="text-red-600 hover:text-red-900 text-xs"
            >
              {actionLoading ? <ButtonLoadingSpinner /> : 'Ban'}
            </button>
          ) : (
            <button
              onClick={() => handleAction('unban')}
              disabled={actionLoading}
              className="text-green-600 hover:text-green-900 text-xs"
            >
              {actionLoading ? <ButtonLoadingSpinner /> : 'Unban'}
            </button>
          )}
          
          {!user.is_admin ? (
            <button
              onClick={() => handleAction('promote')}
              disabled={actionLoading}
              className="text-purple-600 hover:text-purple-900 text-xs"
            >
              {actionLoading ? <ButtonLoadingSpinner /> : 'Promote'}
            </button>
          ) : (
            <button
              onClick={() => handleAction('demote')}
              disabled={actionLoading}
              className="text-orange-600 hover:text-orange-900 text-xs"
            >
              {actionLoading ? <ButtonLoadingSpinner /> : 'Demote'}
            </button>
          )}
        </div>
      </td>
    </tr>
  )
})

UserRow.displayName = 'UserRow'

const LoadingRow = memo(() => (
  <tr>
    <td colSpan={6} className="px-6 py-4 text-center">
      <LoadingSpinner size="sm" />
    </td>
  </tr>
))

LoadingRow.displayName = 'LoadingRow'

const EmptyRow = memo(() => (
  <tr>
    <td colSpan={6} className="px-6 py-12 text-center text-gray-500">
      No users found matching your criteria.
    </td>
  </tr>
))

EmptyRow.displayName = 'EmptyRow'

function AdminUsersTable({ users, loading, hasMore, onLoadMore, onRefresh }: AdminUsersTableProps) {
  const { addToast } = useToast()

  const handleUserAction = async (userId: string, action: string) => {
    try {
      let updateData: any = {}
      let message = ''

      switch (action) {
        case 'ban':
          updateData = { is_banned: true }
          message = 'User banned successfully'
          break
        case 'unban':
          updateData = { is_banned: false }
          message = 'User unbanned successfully'
          break
        case 'promote':
          updateData = { is_admin: true }
          message = 'User promoted to admin'
          break
        case 'demote':
          updateData = { is_admin: false }
          message = 'User demoted from admin'
          break
        default:
          throw new Error(`Unknown action: ${action}`)
      }

      const { error } = await supabase
        .from('profiles')
        .update(updateData)
        .eq('id', userId)

      if (error) {
        throw error
      }

      addToast({
        type: 'success',
        title: 'Action Completed',
        message: message
      })

      onRefresh()

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 'Failed to perform action'
      addToast({
        type: 'error',
        title: 'Action Failed',
        message: errorMessage
      })
    }
  }

  return (
    <div className="bg-white rounded-lg shadow-sm overflow-hidden">
      <div className="overflow-x-auto">
        <table className="min-w-full divide-y divide-gray-200">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">User</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Status</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Performance</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Risk Score</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Activity</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">Actions</th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {users.map((user) => (
              <UserRow 
                key={user.id} 
                user={user} 
                onAction={handleUserAction}
              />
            ))}
            
            {loading && <LoadingRow />}
            
            {!loading && users.length === 0 && <EmptyRow />}
          </tbody>
        </table>
      </div>
      
      {/* Load More button removed since we fetch all users at once */}
    </div>
  )
}

export default memo(AdminUsersTable)
