'use client'

import { useState } from 'react'
import { supabase } from '@/lib/supabase-client'
import { useRouter } from 'next/navigation'

export default function CreateMarketForm() {
  const [loading, setLoading] = useState(false)
  const [message, setMessage] = useState('')
  const router = useRouter()

  const [formData, setFormData] = useState({
    title: '',
    description: '',
    category: 'politics',
    end_date: '',
    image_url: ''
  })

  const categories = [
    { value: 'politics', label: 'Politics' },
    { value: 'sports', label: 'Sports' },
    { value: 'economics', label: 'Economics' },
    { value: 'technology', label: 'Technology' },
    { value: 'entertainment', label: 'Entertainment' },
    { value: 'crypto', label: 'Cryptocurrency' },
    { value: 'weather', label: 'Weather' },
    { value: 'other', label: 'Other' }
  ]

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)
    setMessage('')

    try {
      const { data: { user } } = await supabase.auth.getUser()
      if (!user) {
        throw new Error('You must be logged in to create a market')
      }

      // Validate form data
      if (!formData.title.trim()) {
        throw new Error('Title is required')
      }
      if (!formData.description.trim()) {
        throw new Error('Description is required')
      }
      if (!formData.end_date) {
        throw new Error('End date is required')
      }

      const endDate = new Date(formData.end_date)
      const now = new Date()
      if (endDate <= now) {
        throw new Error('End date must be in the future')
      }

      const { error } = await supabase
        .from('markets')
        .insert({
          title: formData.title.trim(),
          description: formData.description.trim(),
          category: formData.category,
          end_date: endDate.toISOString(),
          image_url: formData.image_url.trim() || null,
          creator_id: user.id,
          status: 'active',
          total_volume: 0,
          yes_price: 0.5,
          no_price: 0.5
        })

      if (error) throw error

      setMessage('Market created successfully!')
      setFormData({
        title: '',
        description: '',
        category: 'politics',
        end_date: '',
        image_url: ''
      })

      // Refresh the page to show the new market
      setTimeout(() => {
        router.refresh()
      }, 1000)

    } catch (error: unknown) {
      setMessage(error instanceof Error ? error.message : 'An error occurred')
    } finally {
      setLoading(false)
    }
  }

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value
    })
  }

  return (
    <div className="max-w-2xl">
      <h3 className="text-lg font-semibold mb-6">Create New Market</h3>
      
      <form onSubmit={handleSubmit} className="space-y-6">
        <div>
          <label htmlFor="title" className="block text-sm font-medium text-black mb-2">
            Market Title *
          </label>
          <input
            id="title"
            name="title"
            type="text"
            value={formData.title}
            onChange={handleChange}
            required
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent !text-black"
            placeholder="e.g., Will Bitcoin reach $100,000 by end of 2024?"
          style={{ color: 'black' }} />
        </div>

        <div>
          <label htmlFor="description" className="block text-sm font-medium text-black mb-2">
            Description *
          </label>
          <textarea
            id="description"
            name="description"
            value={formData.description}
            onChange={handleChange}
            required
            rows={4}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent !text-black"
            placeholder="Provide detailed information about what this market is predicting..."
          style={{ color: 'black' }} />
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <label htmlFor="category" className="block text-sm font-medium text-black mb-2">
              Category *
            </label>
            <select
              id="category"
              name="category"
              value={formData.category}
              onChange={handleChange}
              required
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent !text-black"
             style={{ color: 'black' }}>
              {categories.map((category) => (
                <option key={category.value} value={category.value}>
                  {category.label}
                </option>
              ))}
            </select>
          </div>

          <div>
            <label htmlFor="end_date" className="block text-sm font-medium text-black mb-2">
              End Date *
            </label>
            <input
              id="end_date"
              name="end_date"
              type="datetime-local"
              value={formData.end_date}
              onChange={handleChange}
              required
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent !text-black"
            style={{ color: 'black' }} />
          </div>
        </div>

        <div>
          <label htmlFor="image_url" className="block text-sm font-medium text-black mb-2">
            Image URL (Optional)
          </label>
          <input
            id="image_url"
            name="image_url"
            type="url"
            value={formData.image_url}
            onChange={handleChange}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent !text-black"
            placeholder="https://example.com/image.jpg"
          style={{ color: 'black' }} />
        </div>

        {message && (
          <div className={`p-4 rounded-lg ${
            message.includes('successfully') 
              ? 'bg-green-100 text-green-700 border border-green-200' 
              : 'bg-red-100 text-red-700 border border-red-200'
          }`}>
            {message}
          </div>
        )}

        <button
          type="submit"
          disabled={loading}
          className="w-full bg-blue-600 text-white py-3 px-4 rounded-lg hover:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
        >
          {loading ? 'Creating Market...' : 'Create Market'}
        </button>
      </form>
    </div>
  )
}
