'use client'

import { useState } from 'react'
import { Market } from '@/lib/types'
import { supabase } from '@/lib/supabase-client'

interface MarketsTableProps {
  markets: Market[]
}

export default function MarketsTable({ markets }: MarketsTableProps) {
  const [loading, setLoading] = useState<string | null>(null)

  const handleResolveMarket = async (marketId: string, outcome: 'yes' | 'no') => {
    setLoading(marketId)
    
    try {
      const { error } = await supabase.rpc('resolve_market', {
        market_uuid: marketId,
        resolution: outcome
      })

      if (error) throw error

      // Refresh the page
      window.location.reload()
    } catch (error) {
      console.error('Error resolving market:', error)
      alert('Failed to resolve market')
    } finally {
      setLoading(null)
    }
  }

  const formatPrice = (price: number) => {
    return (price * 100).toFixed(1)
  }

  const formatVolume = (volume: number) => {
    if (volume >= 1000000) {
      return `$${(volume / 1000000).toFixed(1)}M`
    } else if (volume >= 1000) {
      return `$${(volume / 1000).toFixed(1)}K`
    } else {
      return `$${volume.toFixed(0)}`
    }
  }

  const timeRemaining = (endDate: string) => {
    const now = new Date()
    const end = new Date(endDate)
    const diff = end.getTime() - now.getTime()
    
    if (diff <= 0) return 'Ended'
    
    const days = Math.floor(diff / (1000 * 60 * 60 * 24))
    const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60))
    
    if (days > 0) return `${days}d ${hours}h`
    return `${hours}h`
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active':
        return 'bg-green-100 text-green-800'
      case 'closed':
        return 'bg-yellow-100 text-yellow-800'
      case 'resolved':
        return 'bg-blue-100 text-blue-800'
      default:
        return 'bg-gray-100 text-black'
    }
  }

  return (
    <div>
      <div className="flex items-center justify-between mb-6">
        <h3 className="text-lg font-semibold">Markets Management</h3>
        <div className="text-sm text-black">
          {markets.length} total markets
        </div>
      </div>

      <div className="overflow-x-auto">
        <table className="min-w-full divide-y divide-gray-200">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Market
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Category
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Prices
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Volume
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Status
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Time Left
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Actions
              </th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {markets.map((market) => (
              <tr key={market.id} className="hover:bg-gray-50">
                <td className="px-6 py-4 whitespace-nowrap">
                  <div className="flex items-center">
                    <div>
                      <div className="text-sm font-medium text-black max-w-xs truncate">
                        {market.title}
                      </div>
                      <div className="text-sm text-black max-w-xs truncate">
                        {market.description}
                      </div>
                    </div>
                  </div>
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-gray-100 text-black">
                    {market.category}
                  </span>
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                  <div className="flex space-x-2">
                    <span className="text-green-600 font-medium">{formatPrice(market.yes_price || 0.5)}%</span>
                    <span className="text-black">/</span>
                    <span className="text-red-600 font-medium">{formatPrice(market.no_price || 0.5)}%</span>
                  </div>
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                  {formatVolume(market.total_volume)}
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${getStatusColor(market.status)}`}>
                    {market.status}
                  </span>
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                  {timeRemaining(market.end_date)}
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                  {market.status === 'active' && (
                    <div className="flex space-x-2">
                      <button
                        onClick={() => handleResolveMarket(market.id, 'yes')}
                        disabled={loading === market.id}
                        className="bg-green-600 text-white px-3 py-1 rounded text-xs hover:bg-green-700 disabled:opacity-50"
                      >
                        {loading === market.id ? 'Resolving...' : 'Resolve YES'}
                      </button>
                      <button
                        onClick={() => handleResolveMarket(market.id, 'no')}
                        disabled={loading === market.id}
                        className="bg-red-600 text-white px-3 py-1 rounded text-xs hover:bg-red-700 disabled:opacity-50"
                      >
                        Resolve NO
                      </button>
                    </div>
                  )}
                  {market.status === 'resolved' && market.outcome && (
                    <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                      market.outcome === 'yes' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                    }`}>
                      {market.outcome.toUpperCase()}
                    </span>
                  )}
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>

      {markets.length === 0 && (
        <div className="text-center py-12">
          <div className="text-black mb-4">
            <svg className="mx-auto h-12 w-12" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
            </svg>
          </div>
          <h3 className="text-lg font-medium text-black mb-2">No markets found</h3>
          <p className="text-black">Create your first market to get started</p>
        </div>
      )}
    </div>
  )
}
