'use client'

import { useAuth } from './ClientAuthProvider'
import Link from 'next/link'
import { useEffect } from 'react'
import { useRouter } from 'next/navigation'

interface ClientProtectedRouteProps {
  children: React.ReactNode
  requireAdmin?: boolean
}

export default function ClientProtectedRoute({ children, requireAdmin = false }: ClientProtectedRouteProps) {
  const { user, loading, profile, error } = useAuth()
  const router = useRouter()

  useEffect(() => {
    if (!loading && !user) {
      console.log(`[ClientProtectedRoute] No user found, redirecting to login`)
      router.push('/auth/login')
    }
  }, [user, loading, router])

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-black">Loading...</p>
        </div>
      </div>
    )
  }

  if (error) {
    return (
      <div className="min-h-screen bg-red-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-red-800 mb-4">Authentication Error</h1>
          <p className="text-red-600 mb-4">{error}</p>
          <button
            onClick={() => window.location.reload()}
            className="bg-red-600 text-white px-4 py-2 rounded hover:bg-red-700"
          >
            Retry
          </button>
        </div>
      </div>
    )
  }

  if (!user) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Please log in</h1>
          <p className="text-black mb-4">You need to be logged in to access this page.</p>
          <Link href="/auth/login" className="text-blue-600 hover:text-blue-700">
            Go to Login
          </Link>
        </div>
      </div>
    )
  }

  if (requireAdmin) {
    // Wait for profile to load before checking admin status
    if (loading) {
      return (
        <div className="min-h-screen bg-gray-50 flex items-center justify-center">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
            <p className="mt-4 text-black">Checking permissions...</p>
          </div>
        </div>
      )
    }
    
    if (!profile?.is_admin) {
      return (
        <div className="min-h-screen bg-gray-50 flex items-center justify-center">
          <div className="text-center">
            <h1 className="text-2xl font-bold text-black mb-4">Access Denied</h1>
            <p className="text-black mb-4">You don't have permission to access this page.</p>
            <Link href="/" className="text-blue-600 hover:text-blue-700">
              Go Home
            </Link>
          </div>
        </div>
      )
    }
  }

  return <>{children}</>
}