'use client'

import { createContext, useContext, useEffect, useState } from 'react'
import { supabase } from '@/lib/supabase-client'
import { User } from '@supabase/supabase-js'
import { Profile } from '@/lib/types'

interface AuthContextType {
  user: User | null
  profile: Profile | null
  loading: boolean
  error: string | null
  signOut: () => Promise<void>
  refreshProfile: () => Promise<void>
}

const AuthContext = createContext<AuthContextType>({
  user: null,
  profile: null,
  loading: true,
  error: null,
  signOut: async () => {},
  refreshProfile: async () => {},
})

export function FastAuthProvider({ children }: { children: React.ReactNode }) {
  const [user, setUser] = useState<User | null>(null)
  const [profile, setProfile] = useState<Profile | null>(null)
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const fetchProfile = async (userId: string) => {
    try {
      const { data, error } = await supabase
        .from('profiles')
        .select('*')
        .eq('id', userId)
        .single()

      if (error) {
        if (error.code === 'PGRST116') {
          // Create default profile quickly
          const { data: newProfile, error: createError } = await supabase
            .from('profiles')
            .insert({
              id: userId,
              email: user?.email || '',
              username: user?.user_metadata?.username || null,
              is_admin: false,
              balance: 0,
              account_status: 'active',
              kyc_verified: false,
              kyc_status: null,
              total_points: 0,
              created_at: new Date().toISOString(),
              updated_at: new Date().toISOString()
            })
            .select()
            .single()

          if (createError) {
            setProfile(null)
            setError(`Profile creation error: ${createError.message}`)
          } else {
            setProfile(newProfile as Profile)
            setError(null)
          }
        } else {
          setProfile(null)
          setError(`Profile error: ${error.message}`)
        }
      } else {
        setProfile(data as Profile)
        setError(null)
      }
    } catch (err) {
      setProfile(null)
      setError(`Profile fetch error: ${err instanceof Error ? err.message : 'Unknown error'}`)
    }
  }

  const refreshProfile = async () => {
    if (user) {
      await fetchProfile(user.id)
    }
  }

  useEffect(() => {
    let mounted = true

    const initializeAuth = async () => {
      try {
        // Check environment variables
        if (!process.env.NEXT_PUBLIC_SUPABASE_URL || !process.env.NEXT_PUBLIC_SUPABASE_ANON_KEY) {
          setError('Supabase configuration missing')
          setLoading(false)
          return
        }

        const { data: { session }, error } = await supabase.auth.getSession()
        
        if (mounted) {
          if (error) {
            setError(`Session error: ${error.message}`)
            setUser(null)
            setProfile(null)
          } else if (session?.user) {
            setUser(session.user)
            setError(null)
            // Fetch profile in background to not block UI
            fetchProfile(session.user.id)
          } else {
            setUser(null)
            setProfile(null)
            setError(null)
          }
          setLoading(false)
        }
      } catch (error) {
        if (mounted) {
          setUser(null)
          setProfile(null)
          setError(`Auth initialization error: ${error instanceof Error ? error.message : 'Unknown error'}`)
          setLoading(false)
        }
      }
    }

    initializeAuth()

    const { data: { subscription } } = supabase.auth.onAuthStateChange(
      async (event, session) => {
        if (!mounted) return

        setUser(session?.user ?? null)
        
        if (session?.user) {
          // Fetch profile in background
          fetchProfile(session.user.id)
        } else {
          setProfile(null)
          setError(null)
        }
        
        setLoading(false)
      }
    )

    return () => {
      mounted = false
      subscription.unsubscribe()
    }
  }, [])

  const signOut = async () => {
    try {
      await supabase.auth.signOut()
      setUser(null)
      setProfile(null)
      setError(null)
    } catch (error) {
      setError(`Sign out error: ${error instanceof Error ? error.message : 'Unknown error'}`)
    }
  }

  return (
    <AuthContext.Provider value={{ user, profile, loading, error, signOut, refreshProfile }}>
      {children}
    </AuthContext.Provider>
  )
}

export const useAuth = () => {
  const context = useContext(AuthContext)
  if (!context) {
    throw new Error('useAuth must be used within an FastAuthProvider')
  }
  return context
}
