'use client'

import { useAuth } from './FastAuthProvider'
import { useEffect, useState } from 'react'
import { useRouter } from 'next/navigation'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'

interface FastSessionWrapperProps {
  children: React.ReactNode
  requireAuth?: boolean
  requireAdmin?: boolean
  redirectTo?: string
}

export default function FastSessionWrapper({ 
  children, 
  requireAuth = true, 
  requireAdmin = false,
  redirectTo = '/auth/login'
}: FastSessionWrapperProps) {
  const { user, profile, loading, error } = useAuth()
  const router = useRouter()
  const [isRedirecting, setIsRedirecting] = useState(false)

  // Handle authentication redirects
  useEffect(() => {
    if (!loading && requireAuth && !user && !isRedirecting) {
      setIsRedirecting(true)
      router.push(redirectTo)
    }
  }, [user, loading, requireAuth, router, redirectTo, isRedirecting])

  // Handle admin redirects
  useEffect(() => {
    if (!loading && requireAdmin && user && profile && !isRedirecting) {
      if (!profile.is_admin) {
        setIsRedirecting(true)
        router.push('/')
      }
    }
  }, [user, profile, loading, requireAdmin, router, isRedirecting])

  // Show loading state
  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner size="lg" />
          <p className="mt-4 text-black">Loading...</p>
        </div>
      </div>
    )
  }

  // Show error state
  if (error) {
    return (
      <div className="min-h-screen bg-red-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-xl font-bold text-red-800 mb-2">Session Error</h1>
          <p className="text-red-600 mb-4 text-sm">{error}</p>
          <button
            onClick={() => window.location.reload()}
            className="bg-red-600 text-white px-3 py-1 rounded text-sm hover:bg-red-700"
          >
            Retry
          </button>
        </div>
      </div>
    )
  }

  // Show redirecting state
  if (isRedirecting) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <LoadingSpinner size="lg" />
          <p className="mt-4 text-black">Redirecting...</p>
        </div>
      </div>
    )
  }

  // Show authentication required message
  if (requireAuth && !user) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-xl font-bold text-black mb-2">Authentication Required</h1>
          <p className="text-black mb-4 text-sm">You need to be logged in to access this page.</p>
          <button
            onClick={() => router.push('/auth/login')}
            className="bg-blue-600 text-white px-3 py-1 rounded text-sm hover:bg-blue-700"
          >
            Go to Login
          </button>
        </div>
      </div>
    )
  }

  // Show admin required message
  if (requireAdmin && user && profile && !profile.is_admin) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-xl font-bold text-black mb-2">Admin Access Required</h1>
          <p className="text-black mb-4 text-sm">You don't have permission to access this page.</p>
          <button
            onClick={() => router.push('/')}
            className="bg-blue-600 text-white px-3 py-1 rounded text-sm hover:bg-blue-700"
          >
            Go Home
          </button>
        </div>
      </div>
    )
  }

  return <>{children}</>
}
