'use client'

import React, { useEffect } from 'react'
import { useAuth } from '@/components/auth/FastAuthProvider'
import { loginTracking } from '@/lib/services/loginTracking'
import { supabase } from '@/lib/supabase-client'
import { getClientIP } from '@/lib/utils/getClientIP'

interface LoginTrackingWrapperProps {
  children: React.ReactNode
}

export default function LoginTrackingWrapper({ children }: LoginTrackingWrapperProps) {
  const { user } = useAuth()

  useEffect(() => {
    if (user) {
      console.log('LoginTrackingWrapper: User authenticated, starting login tracking...', { userId: user.id })
      
      // Track login session when user is authenticated
      const trackLoginSession = async () => {
        try {
          console.log('LoginTrackingWrapper: Getting session token...')
          // Get current session token
          const { data: { session } } = await supabase.auth.getSession()
          
          if (session?.access_token) {
            console.log('LoginTrackingWrapper: Session token found, tracking login via API...')
            
            // Use API endpoint to track login with proper IP address
            const response = await fetch('/api/track-login', {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify({
                userId: user.id,
                sessionToken: session.access_token
              })
            })
            
            const result = await response.json()
            
            if (result.success) {
              console.log('LoginTrackingWrapper: Login tracked successfully:', result)
            } else {
              console.warn('LoginTrackingWrapper: API tracking failed:', result.error)
              
              // Check if it's a table missing error
              if (result.error?.includes('login_sessions table') || result.error?.includes('does not exist')) {
                console.warn('LoginTrackingWrapper: login_sessions table missing. Please run the fix-login-tracking-complete.sql script.')
                return // Don't try fallback if table doesn't exist
              }
              
              console.warn('LoginTrackingWrapper: Falling back to direct tracking...')
              
              // Fallback to direct tracking with client IP
              try {
                console.log('LoginTrackingWrapper: Getting client IP for fallback...')
                const clientIP = await getClientIP()
                console.log('LoginTrackingWrapper: Client IP:', clientIP)
                const fallbackResult = await loginTracking.trackLogin(user.id, session.access_token, clientIP)
                if (fallbackResult) {
                  console.log('LoginTrackingWrapper: Fallback tracking successful')
                } else {
                  console.warn('LoginTrackingWrapper: Fallback tracking failed - login_sessions table may not exist')
                }
              } catch (fallbackError) {
                console.warn('LoginTrackingWrapper: Fallback tracking also failed:', fallbackError)
              }
            }
          } else {
            console.warn('LoginTrackingWrapper: No session token found')
          }
        } catch (error) {
          console.error('LoginTrackingWrapper: Error tracking login session:', error)
        }
      }

      trackLoginSession()

      // Set up activity tracking
      const activityInterval = setInterval(() => {
        loginTracking.updateActivity()
      }, 5 * 60 * 1000) // Update every 5 minutes

      // Cleanup on unmount
      return () => {
        clearInterval(activityInterval)
        // Track logout when component unmounts
        loginTracking.trackLogout()
      }
    }
  }, [user])

  return <>{children}</>
}
