'use client'

import { useAuth } from './ClientAuthProvider'
import { useEffect, useState } from 'react'
import { useRouter } from 'next/navigation'
import { supabase } from '@/lib/supabase-client'
import { syncSessionWithServer } from '@/lib/session-sync'

interface SessionWrapperProps {
  children: React.ReactNode
  requireAuth?: boolean
  requireAdmin?: boolean
  redirectTo?: string
}

export default function SessionWrapper({ 
  children, 
  requireAuth = true, 
  requireAdmin = false,
  redirectTo = '/auth/login'
}: SessionWrapperProps) {
  const { user, profile, loading, error } = useAuth()
  const router = useRouter()
  const [sessionSynced, setSessionSynced] = useState(false)
  const [sessionError, setSessionError] = useState<string | null>(null)

  // Sync session with server
  useEffect(() => {
    const syncSession = async () => {
      if (user && !sessionSynced) {
        try {
          console.log('[SessionWrapper] Syncing session with server for user:', user.email)
          const synced = await syncSessionWithServer()
          if (synced) {
            console.log('[SessionWrapper] Session synced successfully')
            setSessionSynced(true)
            setSessionError(null)
          } else {
            console.warn('[SessionWrapper] Session sync failed, but continuing with client session')
            // Don't set error, just mark as synced to continue
            setSessionSynced(true)
            setSessionError(null)
          }
        } catch (err) {
          console.warn('[SessionWrapper] Session sync error, but continuing with client session:', err)
          // Don't set error, just mark as synced to continue
          setSessionSynced(true)
          setSessionError(null)
        }
      }
    }

    syncSession()
  }, [user, sessionSynced])

  // Handle authentication redirects
  useEffect(() => {
    if (!loading && requireAuth && !user) {
      console.log('[SessionWrapper] No user found, redirecting to:', redirectTo)
      router.push(redirectTo)
    }
  }, [user, loading, requireAuth, router, redirectTo])

  // Handle admin redirects
  useEffect(() => {
    if (!loading && requireAdmin && user && profile) {
      if (!profile.is_admin) {
        console.log('[SessionWrapper] User is not admin, redirecting to home')
        router.push('/')
      }
    }
  }, [user, profile, loading, requireAdmin, router])

  // Show loading state
  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-black">Loading session...</p>
        </div>
      </div>
    )
  }

  // Show error state
  if (error || sessionError) {
    return (
      <div className="min-h-screen bg-red-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-red-800 mb-4">Session Error</h1>
          <p className="text-red-600 mb-4">{error || sessionError}</p>
          <button
            onClick={() => window.location.reload()}
            className="bg-red-600 text-white px-4 py-2 rounded hover:bg-red-700"
          >
            Retry
          </button>
        </div>
      </div>
    )
  }

  // Show authentication required message
  if (requireAuth && !user) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Authentication Required</h1>
          <p className="text-black mb-4">You need to be logged in to access this page.</p>
          <button
            onClick={() => router.push('/auth/login')}
            className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Go to Login
          </button>
        </div>
      </div>
    )
  }

  // Show admin required message
  if (requireAdmin && user && profile && !profile.is_admin) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Admin Access Required</h1>
          <p className="text-black mb-4">You don't have permission to access this page.</p>
          <button
            onClick={() => router.push('/')}
            className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Go Home
          </button>
        </div>
      </div>
    )
  }

  // Show session sync loading
  if (user && !sessionSynced) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-black">Synchronizing session...</p>
        </div>
      </div>
    )
  }

  return <>{children}</>
}
