'use client'

import { createContext, useContext, useEffect, useState } from 'react'
import { supabase } from '@/lib/supabase-client'
import { User } from '@supabase/supabase-js'
import { Profile } from '@/lib/types'

interface AuthContextType {
  user: User | null
  profile: Profile | null
  loading: boolean
  error: string | null
  signOut: () => Promise<void>
  refreshProfile: () => Promise<void>
}

const AuthContext = createContext<AuthContextType>({
  user: null,
  profile: null,
  loading: true,
  error: null,
  signOut: async () => {},
  refreshProfile: async () => {},
})

export function StableAuthProvider({ children }: { children: React.ReactNode }) {
  const [user, setUser] = useState<User | null>(null)
  const [profile, setProfile] = useState<Profile | null>(null)
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const fetchProfile = async (userId: string) => {
    try {
      console.log(`[StableAuthProvider] Fetching profile for user: ${userId}`)
      const { data, error } = await supabase
        .from('profiles')
        .select('*')
        .eq('id', userId)
        .single()

      if (error) {
        console.error(`[StableAuthProvider] Error fetching profile:`, error)
        // Don't set error for missing profile - user might not have one yet
        if (error.code === 'PGRST116') {
          console.log(`[StableAuthProvider] No profile found for user ${userId}, creating default profile`)
          // Create a default profile
          const { data: newProfile, error: createError } = await supabase
            .from('profiles')
            .insert({
              id: userId,
              email: user?.email || '',
              username: user?.user_metadata?.username || null,
              is_admin: false,
              balance: 0,
              account_status: 'active',
              kyc_verified: false,
              kyc_status: null,
              total_points: 0,
              created_at: new Date().toISOString(),
              updated_at: new Date().toISOString()
            })
            .select()
            .single()

          if (createError) {
            console.error(`[StableAuthProvider] Error creating profile:`, createError)
            setProfile(null)
            setError(`Profile creation error: ${createError.message}`)
          } else {
            console.log(`[StableAuthProvider] Default profile created:`, newProfile)
            setProfile(newProfile as Profile)
            setError(null)
          }
        } else {
          setProfile(null)
          setError(`Profile error: ${error.message}`)
        }
      } else {
        console.log(`[StableAuthProvider] Profile fetched successfully:`, data)
        setProfile(data as Profile)
        setError(null)
      }
    } catch (err) {
      console.error(`[StableAuthProvider] Error fetching profile:`, err)
      setProfile(null)
      setError(`Profile fetch error: ${err instanceof Error ? err.message : 'Unknown error'}`)
    }
  }

  const refreshProfile = async () => {
    if (user) {
      await fetchProfile(user.id)
    }
  }

  useEffect(() => {
    let mounted = true

    const initializeAuth = async () => {
      try {
        console.log(`[StableAuthProvider] Initializing authentication...`)
        
        // Check environment variables
        if (!process.env.NEXT_PUBLIC_SUPABASE_URL || !process.env.NEXT_PUBLIC_SUPABASE_ANON_KEY) {
          console.error(`[StableAuthProvider] Missing Supabase configuration`)
          setError('Supabase configuration missing')
          setLoading(false)
          return
        }

        const { data: { session }, error } = await supabase.auth.getSession()
        
        if (mounted) {
          if (error) {
            console.error(`[StableAuthProvider] Session error:`, error)
            setError(`Session error: ${error.message}`)
            setUser(null)
            setProfile(null)
          } else if (session?.user) {
            console.log(`[StableAuthProvider] Session found for user: ${session.user.email}`)
            setUser(session.user)
            setError(null)
            await fetchProfile(session.user.id)
          } else {
            console.log(`[StableAuthProvider] No session found`)
            setUser(null)
            setProfile(null)
            setError(null)
          }
          setLoading(false)
        }
      } catch (error) {
        console.error(`[StableAuthProvider] Auth initialization error:`, error)
        if (mounted) {
          setUser(null)
          setProfile(null)
          setError(`Auth initialization error: ${error instanceof Error ? error.message : 'Unknown error'}`)
          setLoading(false)
        }
      }
    }

    initializeAuth()

    const { data: { subscription } } = supabase.auth.onAuthStateChange(
      async (event, session) => {
        if (!mounted) return

        console.log(`[StableAuthProvider] Auth state change: ${event}`)
        console.log(`[StableAuthProvider] Session: ${session ? 'Valid' : 'None'}`)
        console.log(`[StableAuthProvider] User: ${session?.user?.email || 'None'}`)
        
        setUser(session?.user ?? null)
        
        if (session?.user) {
          console.log(`[StableAuthProvider] Fetching profile for user: ${session.user.email}`)
          await fetchProfile(session.user.id)
          // NO session sync - this was causing the refresh failures
        } else {
          console.log(`[StableAuthProvider] No session, clearing profile`)
          setProfile(null)
          setError(null)
        }
        
        setLoading(false)
      }
    )

    return () => {
      console.log(`[StableAuthProvider] Cleaning up auth subscription`)
      mounted = false
      subscription.unsubscribe()
    }
  }, [])

  const signOut = async () => {
    try {
      console.log(`[StableAuthProvider] Signing out user`)
      await supabase.auth.signOut()
      setUser(null)
      setProfile(null)
      setError(null)
    } catch (error) {
      console.error(`[StableAuthProvider] Error signing out:`, error)
      setError(`Sign out error: ${error instanceof Error ? error.message : 'Unknown error'}`)
    }
  }

  return (
    <AuthContext.Provider value={{ user, profile, loading, error, signOut, refreshProfile }}>
      {children}
    </AuthContext.Provider>
  )
}

export const useAuth = () => {
  const context = useContext(AuthContext)
  if (!context) {
    throw new Error('useAuth must be used within an AuthProvider')
  }
  return context
}
