'use client'

import { useAuth } from './StableAuthProvider'
import { useEffect } from 'react'
import { useRouter } from 'next/navigation'

interface StableSessionWrapperProps {
  children: React.ReactNode
  requireAuth?: boolean
  requireAdmin?: boolean
  redirectTo?: string
}

export default function StableSessionWrapper({ 
  children, 
  requireAuth = true, 
  requireAdmin = false,
  redirectTo = '/auth/login'
}: StableSessionWrapperProps) {
  const { user, profile, loading, error } = useAuth()
  const router = useRouter()

  // Handle authentication redirects
  useEffect(() => {
    if (!loading && requireAuth && !user) {
      console.log('[StableSessionWrapper] No user found, redirecting to:', redirectTo)
      router.push(redirectTo)
    }
  }, [user, loading, requireAuth, router, redirectTo])

  // Handle admin redirects
  useEffect(() => {
    if (!loading && requireAdmin && user && profile) {
      if (!profile.is_admin) {
        console.log('[StableSessionWrapper] User is not admin, redirecting to home')
        router.push('/')
      }
    }
  }, [user, profile, loading, requireAdmin, router])

  // Show loading state
  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-black">Loading session...</p>
        </div>
      </div>
    )
  }

  // Show error state
  if (error) {
    return (
      <div className="min-h-screen bg-red-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-red-800 mb-4">Session Error</h1>
          <p className="text-red-600 mb-4">{error}</p>
          <button
            onClick={() => window.location.reload()}
            className="bg-red-600 text-white px-4 py-2 rounded hover:bg-red-700"
          >
            Retry
          </button>
        </div>
      </div>
    )
  }

  // Show authentication required message
  if (requireAuth && !user) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Authentication Required</h1>
          <p className="text-black mb-4">You need to be logged in to access this page.</p>
          <button
            onClick={() => router.push('/auth/login')}
            className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Go to Login
          </button>
        </div>
      </div>
    )
  }

  // Show admin required message
  if (requireAdmin && user && profile && !profile.is_admin) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-black mb-4">Admin Access Required</h1>
          <p className="text-black mb-4">You don't have permission to access this page.</p>
          <button
            onClick={() => router.push('/')}
            className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Go Home
          </button>
        </div>
      </div>
    )
  }

  return <>{children}</>
}
