'use client'

import { useState, useEffect } from 'react'
import Link from 'next/link'
import { useRouter } from 'next/navigation'
import { useAuth } from '@/components/auth/FastAuthProvider'
import LoadingLink from '@/components/ui/LoadingLink'
import { useProfile } from '@/lib/hooks/useProfile'
import { supabase } from '@/lib/supabase-client'
import LiveBalance from './LiveBalance'
import { useNotifications } from '@/lib/hooks/useNotifications'

export default function Navigation() {
  const { user, loading, signOut } = useAuth()
  const { profile } = useProfile(user?.id)
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false)
  const [profileDropdownOpen, setProfileDropdownOpen] = useState(false)
  const [searchQuery, setSearchQuery] = useState('')
  const [notificationsOpen, setNotificationsOpen] = useState(false)
  
  // Use the new notification hook
  const { 
    notifications, 
    unreadCount, 
    markAsRead, 
    markAllAsRead 
  } = useNotifications(user?.id)
  const [searchField, setSearchField] = useState<'all' | 'title' | 'description' | 'category' | 'status'>('all')
  const [suggestions, setSuggestions] = useState<any[]>([])
  const [showSuggestions, setShowSuggestions] = useState(false)
  const [activeSuggestionIndex, setActiveSuggestionIndex] = useState(-1)
  const router = useRouter()

  const handleSignOut = async () => {
    await signOut()
    window.location.href = '/auth/login'
  }

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault()
    if (searchQuery.trim()) {
      // Navigate to home and let user refine filters there for now
      router.push('/')
      setShowSuggestions(false)
    }
  }

  // Debounced suggestions fetch
  useEffect(() => {
    const q = searchQuery.trim()
    if (!q) {
      setSuggestions([])
      return
    }

    const controller = new AbortController()
    const timer = setTimeout(async () => {
      try {
        // Build query
        let query = supabase
          .from('markets')
          .select('id,title,category,status')
          .or('status.eq.active,status.eq.pending')
          .limit(8)

        const pattern = `%${q}%`
        if (searchField === 'title') {
          query = query.ilike('title', pattern)
        } else if (searchField === 'description') {
          query = query.ilike('description', pattern)
        } else if (searchField === 'category') {
          query = query.ilike('category', pattern)
        } else if (searchField === 'status') {
          query = query.ilike('status', pattern)
        } else {
          // all
          query = query.or(`title.ilike.${pattern},description.ilike.${pattern},category.ilike.${pattern}`)
        }

        const { data, error } = await query
        if (!error && data) {
          setSuggestions(data)
        }
      } catch (err) {
        // ignore
      }
    }, 250)

    return () => {
      controller.abort()
      clearTimeout(timer)
    }
  }, [searchQuery, searchField])

  // Notification handling is now managed by the useNotifications hook

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  if (loading) {
    return (
      <header className="bg-white shadow-sm border-b sticky top-0 z-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between h-16">
            <div className="flex items-center space-x-3">
              <div className="w-8 h-8 sm:w-10 sm:h-10 bg-blue-600 rounded-lg flex items-center justify-center">
                <span className="text-white font-bold text-lg sm:text-xl">B</span>
              </div>
              <span className="text-xl sm:text-2xl font-bold text-black">BetFusion Zim</span>
            </div>
            <div className="animate-pulse bg-gray-200 h-8 w-20 rounded"></div>
          </div>
        </div>
      </header>
    )
  }

  return (
    <header className="bg-white shadow-sm border-b sticky top-0 z-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex items-center justify-between h-16">
          {/* Logo */}
          <div className="flex items-center space-x-2 sm:space-x-3">
            <Link href="/" className="flex items-center space-x-2 sm:space-x-3">
              <div className="w-8 h-8 sm:w-10 sm:h-10 bg-blue-600 rounded-lg flex items-center justify-center">
                <span className="text-white font-bold text-lg sm:text-xl">B</span>
              </div>
              <span className="text-xl sm:text-2xl font-bold text-black">BetFusion Zim</span>
            </Link>
          </div>

          {/* Center Search Bar - Desktop Only */}
          <div className="hidden md:flex flex-1 max-w-xl mx-8">
            <form onSubmit={handleSearch} className="w-full">
              <div className="relative">
                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                  <svg className="h-5 w-5 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                  </svg>
                </div>
                <input
                  type="text"
                  value={searchQuery}
                  onChange={(e) => {
                    setSearchQuery(e.target.value)
                    setShowSuggestions(true)
                    setActiveSuggestionIndex(-1)
                  }}
                  onFocus={() => setShowSuggestions(true)}
                  onBlur={() => setTimeout(() => setShowSuggestions(false), 120)}
                  onKeyDown={(e) => {
                    if (!showSuggestions || suggestions.length === 0) return
                    if (e.key === 'ArrowDown') {
                      e.preventDefault()
                      setActiveSuggestionIndex((prev) => Math.min(prev + 1, suggestions.length - 1))
                    } else if (e.key === 'ArrowUp') {
                      e.preventDefault()
                      setActiveSuggestionIndex((prev) => Math.max(prev - 1, 0))
                    } else if (e.key === 'Enter' && activeSuggestionIndex >= 0) {
                      e.preventDefault()
                      const sel = suggestions[activeSuggestionIndex]
                      router.push(`/markets/${sel.id}`)
                      setShowSuggestions(false)
                    }
                  }}
                  placeholder="Search BetFusion Zim"
                  className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-gray-50"
                />
                <div className="absolute inset-y-0 right-0 pr-3 flex items-center">
                  {/* Field filter */}
                  <select
                    value={searchField}
                    onChange={(e) => setSearchField(e.target.value as any)}
                    className="hidden lg:block text-xs px-2 py-1 border border-gray-300 rounded bg-white text-black focus:outline-none focus:ring-1 focus:ring-blue-500"
                    title="Filter field"
                  >
                    <option value="all">All</option>
                    <option value="title">Title</option>
                    <option value="description">Description</option>
                    <option value="category">Category</option>
                    <option value="status">Status</option>
                  </select>
                </div>
                {/* Suggestions dropdown */}
                {showSuggestions && suggestions.length > 0 && (
                  <div className="absolute z-50 mt-1 w-full bg-white border border-gray-200 rounded-md shadow-lg max-h-80 overflow-auto">
                    {suggestions.map((s, idx) => (
                      <button
                        key={s.id}
                        type="button"
                        onMouseDown={(e) => {
                          e.preventDefault()
                          router.push(`/markets/${s.id}`)
                          setShowSuggestions(false)
                        }}
                        className={`w-full text-left px-3 py-2 text-sm ${idx === activeSuggestionIndex ? 'bg-blue-50' : 'bg-white'} text-black hover:bg-blue-50`}
                      >
                        <div className="flex items-center justify-between">
                          <span className="truncate pr-2">{s.title}</span>
                          <span className="ml-2 text-xs text-gray-600 capitalize">{s.category}</span>
                        </div>
                      </button>
                    ))}
                  </div>
                )}
              </div>
            </form>
          </div>

          {/* Right side - Auth or User menu */}
          <div className="flex items-center space-x-2 sm:space-x-4">
            {user ? (
              <>
                {/* Live Balance - Desktop Only */}
                <div className="hidden lg:flex">
                  <LiveBalance />
                </div>

                {/* Notifications Icon */}
                <div className="relative">
                  <button 
                    onClick={() => {
                      setNotificationsOpen(!notificationsOpen)
                      setProfileDropdownOpen(false)
                    }}
                    className="relative p-2 rounded-lg hover:bg-gray-50 transition-colors"
                  >
                    <svg className="w-6 h-6 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M11 5.882V19.24a1.76 1.76 0 01-3.417.592l-2.147-6.15M18 13a3 3 0 100-6M5.436 13.683A4.001 4.001 0 017 6h1.832c4.1 0 7.625-1.234 9.168-3v14c-1.543-1.766-5.067-3-9.168-3H7a3.988 3.988 0 01-1.564-.317z" />
                    </svg>
                    {/* Notification badge */}
                    {unreadCount > 0 && (
                      <span className="absolute top-1 right-1 w-2 h-2 bg-red-500 rounded-full"></span>
                    )}
                  </button>

                  {/* Notifications Dropdown */}
                  {notificationsOpen && (
                    <div className="absolute right-0 mt-2 w-80 bg-white rounded-lg shadow-lg border border-gray-200 py-2 z-50">
                      <div className="px-4 py-3 border-b border-gray-100">
                        <h3 className="text-sm font-semibold text-black">Notifications</h3>
                      </div>
                      
                      <div className="max-h-64 overflow-y-auto">
                        {notifications.length > 0 ? (
                          notifications.slice(0, 5).map((notification) => (
                            <div 
                              key={notification.id} 
                              className={`px-4 py-3 hover:bg-gray-50 cursor-pointer border-b border-gray-100 last:border-b-0 ${
                                !notification.is_read ? 'bg-blue-50' : ''
                              }`}
                              onClick={() => markAsRead(notification.id)}
                            >
                              <div className="flex items-start space-x-3">
                                <div className="flex-shrink-0">
                                  <div className={`w-2 h-2 rounded-full mt-2 ${
                                    notification.type === 'success' ? 'bg-green-500' :
                                    notification.type === 'error' ? 'bg-red-500' :
                                    notification.type === 'warning' ? 'bg-yellow-500' :
                                    'bg-blue-500'
                                  }`}></div>
                                </div>
                                <div className="flex-1 min-w-0">
                                  <p className="text-sm font-medium text-black">{notification.title}</p>
                                  <p className="text-xs text-gray-600 mt-1">{notification.message}</p>
                                  <p className="text-xs text-gray-500 mt-1">
                                    {new Date(notification.created_at).toLocaleTimeString()}
                                  </p>
                                </div>
                              </div>
                            </div>
                          ))
                        ) : (
                          <div className="px-4 py-8 text-center">
                            <div className="text-black mb-2">No new notifications</div>
                            <div className="text-xs text-black">You're all caught up!</div>
                          </div>
                        )}
                      </div>

                      {notifications.length > 0 && (
                        <div className="px-4 py-2 border-t border-gray-100">
                          <Link 
                            href="/notifications" 
                            className="text-sm text-blue-600 hover:text-blue-700"
                            onClick={() => setNotificationsOpen(false)}
                          >
                            View all notifications
                          </Link>
                        </div>
                      )}
                    </div>
                  )}
                </div>

                {/* Profile Dropdown */}
                <div className="relative">
                  <button
                    onClick={() => setProfileDropdownOpen(!profileDropdownOpen)}
                    className="flex items-center space-x-2 p-1 rounded-lg hover:bg-gray-50 transition-colors"
                  >
                    <div className="w-8 h-8 bg-gradient-to-br from-pink-500 via-purple-500 to-white rounded-full flex items-center justify-center overflow-hidden">
                      {profile?.avatar_url ? (
                        <img 
                          src={profile.avatar_url} 
                          alt="Profile" 
                          className="w-full h-full object-cover"
                        />
                      ) : (
                        <span className="text-white text-sm font-bold">
                          {(profile?.username || user.email)?.charAt(0).toUpperCase()}
                        </span>
                      )}
                    </div>
                    <svg className="w-4 h-4 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                    </svg>
                  </button>

                  {/* Profile Dropdown Menu */}
                  {profileDropdownOpen && (
                    <div className="absolute right-0 mt-2 w-64 bg-white rounded-lg shadow-lg border border-gray-200 py-2 z-50">
                      <div className="px-4 py-3 border-b border-gray-100">
                        <div className="flex items-center space-x-3">
                          <div className="w-10 h-10 bg-gradient-to-br from-pink-500 via-purple-500 to-white rounded-full flex items-center justify-center overflow-hidden">
                            {profile?.avatar_url ? (
                              <img 
                                src={profile.avatar_url} 
                                alt="Profile" 
                                className="w-full h-full object-cover"
                              />
                            ) : (
                              <span className="text-white font-bold">
                                {(profile?.username || user.email)?.charAt(0).toUpperCase()}
                              </span>
                            )}
                          </div>
                          <div>
                            <div className="font-medium text-black">{profile?.username || 'User'}</div>
                            <div className="text-sm text-black">{user.email}</div>
                          </div>
                        </div>
                      </div>
                      
                      <div className="py-2">
                        <Link 
                          href="/profile" 
                          className="flex items-center px-4 py-2 text-sm text-black hover:bg-gray-50"
                          onClick={() => setProfileDropdownOpen(false)}
                        >
                          <svg className="w-4 h-4 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                          </svg>
                          Profile
                        </Link>
                        <Link 
                          href="/portfolio" 
                          className="flex items-center px-4 py-2 text-sm text-black hover:bg-gray-50"
                          onClick={() => setProfileDropdownOpen(false)}
                        >
                          <svg className="w-4 h-4 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1" />
                          </svg>
                          Portfolio
                        </Link>
                        <Link 
                          href="/wallet" 
                          className="flex items-center px-4 py-2 text-sm text-black hover:bg-gray-50"
                          onClick={() => setProfileDropdownOpen(false)}
                        >
                          <svg className="w-4 h-4 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z" />
                          </svg>
                          Wallet
                        </Link>
                        <Link 
                          href="/leaderboard" 
                          className="flex items-center px-4 py-2 text-sm text-black hover:bg-gray-50"
                          onClick={() => setProfileDropdownOpen(false)}
                        >
                          <svg className="w-4 h-4 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z" />
                          </svg>
                          Leaderboard
                        </Link>
                        <Link 
                          href="/rewards" 
                          className="flex items-center px-4 py-2 text-sm text-black hover:bg-gray-50"
                          onClick={() => setProfileDropdownOpen(false)}
                        >
                          <svg className="w-4 h-4 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M11.049 2.927c.3-.921 1.603-.921 1.902 0l1.519 4.674a1 1 0 00.95.69h4.915c.969 0 1.371 1.24.588 1.81l-3.976 2.888a1 1 0 00-.363 1.118l1.518 4.674c.3.922-.755 1.688-1.538 1.118l-3.976-2.888a1 1 0 00-1.176 0l-3.976 2.888c-.783.57-1.838-.197-1.538-1.118l1.518-4.674a1 1 0 00-.363-1.118l-3.976-2.888c-.784-.57-.38-1.81.588-1.81h4.914a1 1 0 00.951-.69l1.519-4.674z" />
                          </svg>
                          Rewards
                        </Link>
                        <Link 
                          href="/watchlist" 
                          className="flex items-center px-4 py-2 text-sm text-black hover:bg-gray-50"
                          onClick={() => setProfileDropdownOpen(false)}
                        >
                          <svg className="w-4 h-4 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z" />
                          </svg>
                          Watchlist
                        </Link>
                        <LoadingLink 
                          href="/create-market" 
                          className="flex items-center px-4 py-2 text-sm text-black hover:bg-gray-50"
                          onClick={() => setProfileDropdownOpen(false)}
                        >
                          <svg className="w-4 h-4 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
                          </svg>
                          Create Market
                        </LoadingLink>
                        <Link 
                          href="/kyc" 
                          className="flex items-center px-4 py-2 text-sm text-black hover:bg-gray-50"
                          onClick={() => setProfileDropdownOpen(false)}
                        >
                          <svg className="w-4 h-4 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                          </svg>
                          KYC Verification
                        </Link>
                        <Link 
                          href="/settings" 
                          className="flex items-center px-4 py-2 text-sm text-black hover:bg-gray-50"
                          onClick={() => setProfileDropdownOpen(false)}
                        >
                          <svg className="w-4 h-4 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0l1.403 5.759c.2.821.8 1.47 1.609 1.739l5.478 1.593c1.761.512 1.761 2.954 0 3.466l-5.478 1.593c-.809.269-1.409.918-1.609 1.739l-1.403 5.759c-.426 1.756-2.924 1.756-3.35 0l-1.403-5.759c-.2-.821-.8-1.47-1.609-1.739l-5.478-1.593c-1.761-.512-1.761-2.954 0-3.466l5.478-1.593c.809-.269 1.409-.918 1.609-1.739l1.403-5.759z" />
                          </svg>
                          Settings
                        </Link>
                      </div>
                      
                      <div className="border-t border-gray-100 py-2">
                        <button 
                          onClick={handleSignOut}
                          className="flex items-center w-full px-4 py-2 text-sm text-red-600 hover:bg-red-50"
                        >
                          <svg className="w-4 h-4 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1" />
                          </svg>
                          Sign Out
                        </button>
                      </div>
                    </div>
                  )}
                </div>

                {/* Mobile Menu Button */}
                <button
                  onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
                  className="md:hidden p-2 rounded-lg hover:bg-gray-50 transition-colors"
                >
                  <svg className="w-6 h-6 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    {mobileMenuOpen ? (
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                    ) : (
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                    )}
                  </svg>
                </button>
              </>
            ) : (
              <div className="flex items-center space-x-2">
                <Link 
                  href="/auth/login" 
                  className="text-sm sm:text-base font-medium text-black hover:text-blue-600 transition-colors"
                >
                  Sign In
                </Link>
                <Link 
                  href="/auth/signup" 
                  className="bg-blue-600 text-white px-3 py-2 sm:px-4 sm:py-2 rounded-lg font-medium hover:bg-blue-700 transition-colors text-sm sm:text-base"
                >
                  Sign Up
                </Link>
              </div>
            )}
          </div>
        </div>

        {/* Mobile Search Bar */}
        {user && (
          <div className="md:hidden border-t border-gray-200 px-4 py-3">
            <form onSubmit={handleSearch}>
              <div className="relative">
                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                  <svg className="h-5 w-5 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                  </svg>
                </div>
                <input
                  type="text"
                  value={searchQuery}
                  onChange={(e) => {
                    setSearchQuery(e.target.value)
                    setShowSuggestions(true)
                    setActiveSuggestionIndex(-1)
                  }}
                  onFocus={() => setShowSuggestions(true)}
                  onBlur={() => setTimeout(() => setShowSuggestions(false), 120)}
                  onKeyDown={(e) => {
                    if (!showSuggestions || suggestions.length === 0) return
                    if (e.key === 'ArrowDown') {
                      e.preventDefault()
                      setActiveSuggestionIndex((prev) => Math.min(prev + 1, suggestions.length - 1))
                    } else if (e.key === 'ArrowUp') {
                      e.preventDefault()
                      setActiveSuggestionIndex((prev) => Math.max(prev - 1, 0))
                    } else if (e.key === 'Enter' && activeSuggestionIndex >= 0) {
                      e.preventDefault()
                      const sel = suggestions[activeSuggestionIndex]
                      router.push(`/markets/${sel.id}`)
                      setShowSuggestions(false)
                    }
                  }}
                  placeholder="Search BetFusion Zim"
                  className="w-full pl-10 pr-20 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-gray-50"
                />
                {/* Field filter on the right */}
                <div className="absolute inset-y-0 right-0 pr-2 flex items-center">
                  <select
                    value={searchField}
                    onChange={(e) => setSearchField(e.target.value as any)}
                    className="text-xs px-2 py-1 border border-gray-300 rounded bg-white text-black focus:outline-none focus:ring-1 focus:ring-blue-500"
                    title="Filter field"
                  >
                    <option value="all">All</option>
                    <option value="title">Title</option>
                    <option value="description">Description</option>
                    <option value="category">Category</option>
                    <option value="status">Status</option>
                  </select>
                </div>

                {/* Suggestions dropdown */}
                {showSuggestions && suggestions.length > 0 && (
                  <div className="absolute z-50 mt-1 w-full bg-white border border-gray-200 rounded-md shadow-lg max-h-80 overflow-auto">
                    {suggestions.map((s, idx) => (
                      <button
                        key={s.id}
                        type="button"
                        onMouseDown={(e) => {
                          e.preventDefault()
                          router.push(`/markets/${s.id}`)
                          setShowSuggestions(false)
                        }}
                        className={`w-full text-left px-3 py-2 text-sm ${idx === activeSuggestionIndex ? 'bg-blue-50' : 'bg-white'} text-black hover:bg-blue-50`}
                      >
                        <div className="flex items-center justify-between">
                          <span className="truncate pr-2">{s.title}</span>
                          <span className="ml-2 text-xs text-gray-600 capitalize">{s.category}</span>
                        </div>
                      </button>
                    ))}
                  </div>
                )}
              </div>
            </form>
          </div>
        )}

        {/* Mobile Menu */}
        {mobileMenuOpen && user && (
          <div className="md:hidden border-t border-gray-200 bg-white">
            <div className="px-4 py-4">
              {/* Live Balance for Mobile */}
              <div className="mb-4 lg:hidden">
                <LiveBalance />
              </div>
              <div className="space-y-2">
              <Link 
                href="/" 
                className="flex items-center px-3 py-2 text-base font-medium text-black hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                onClick={() => setMobileMenuOpen(false)}
              >
                Markets
              </Link>
              <Link 
                href="/portfolio" 
                className="flex items-center px-3 py-2 text-base font-medium text-black hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                onClick={() => setMobileMenuOpen(false)}
              >
                Portfolio
              </Link>
              <Link 
                href="/wallet" 
                className="flex items-center px-3 py-2 text-base font-medium text-black hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                onClick={() => setMobileMenuOpen(false)}
              >
                Wallet
              </Link>
              <Link 
                href="/rewards" 
                className="flex items-center px-3 py-2 text-base font-medium text-black hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                onClick={() => setMobileMenuOpen(false)}
              >
                Rewards
              </Link>
              <Link 
                href="/leaderboard" 
                className="flex items-center px-3 py-2 text-base font-medium text-black hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                onClick={() => setMobileMenuOpen(false)}
              >
                Leaderboard
              </Link>
              <Link 
                href="/profile" 
                className="flex items-center px-3 py-2 text-base font-medium text-black hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                onClick={() => setMobileMenuOpen(false)}
              >
                Profile
              </Link>
              <Link 
                href="/kyc" 
                className="flex items-center px-3 py-2 text-base font-medium text-black hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                onClick={() => setMobileMenuOpen(false)}
              >
                KYC Verification
              </Link>
              <Link 
                href="/settings" 
                className="flex items-center px-3 py-2 text-base font-medium text-black hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                onClick={() => setMobileMenuOpen(false)}
              >
                Settings
              </Link>
              <hr className="my-2" />
              <button 
                onClick={() => {
                  handleSignOut()
                  setMobileMenuOpen(false)
                }}
                className="flex items-center w-full px-3 py-2 text-base font-medium text-red-600 hover:bg-red-50 rounded-lg transition-colors"
              >
                Sign Out
              </button>
              </div>
            </div>
          </div>
        )}
      </div>

      {/* Close dropdowns when clicking outside */}
      {(profileDropdownOpen || notificationsOpen) && (
        <div 
          className="fixed inset-0 z-10" 
          onClick={() => {
            setProfileDropdownOpen(false)
            setNotificationsOpen(false)
          }}
        />
      )}
    </header>
  )
}