'use client'

import { useState } from 'react'
import { Market } from '@/lib/types'
import { supabase } from '@/lib/supabase-client'
import { BettingEngine } from '@/lib/betting'
import { LoadingSpinner, ButtonLoadingSpinner, BetFormLoadingSpinner } from '@/components/ui/LoadingSpinner'
import { useToast } from '@/components/ui/Toast'

interface BetFormProps {
  market: Market
  onBetPlaced: () => void
}

export default function BetForm({ market, onBetPlaced }: BetFormProps) {
  const [outcome, setOutcome] = useState<'yes' | 'no'>('yes')
  const [amount, setAmount] = useState('')
  const [seededAmount, setSeededAmount] = useState('')
  const [betType, setBetType] = useState<'live' | 'seed'>('live')
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState('')
  const [success, setSuccess] = useState(false)
  const { addToast } = useToast()

  // Automatically switch to live betting if market is already live and seeding is selected
  const isMarketLive = (market.live_volume || 0) > 0
  const effectiveBetType = isMarketLive && betType === 'seed' ? 'live' : betType

  const currentPrice = outcome === 'yes' ? (market.yes_price || 0.5) : (market.no_price || 0.5)
  const shares = amount ? (parseFloat(amount) / currentPrice).toFixed(2) : '0.00'

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)
    setError('')

    try {
      const { data: { user } } = await supabase.auth.getUser()
      if (!user) {
        throw new Error('You must be logged in to place a bet')
      }

      if (effectiveBetType === 'live') {
        const betAmount = parseFloat(amount)
        if (betAmount <= 0) {
          throw new Error('Amount must be greater than 0')
        }

        // Check if market is still active
        if (market.status !== 'active') {
          throw new Error('This market is no longer accepting bets')
        }

        // Use BettingEngine to place share-based bet
        const result = await BettingEngine.placeBet(
          user.id,
          market.id,
          outcome,
          betAmount
        )

        if (!result.success) {
          throw new Error(result.error)
        }

        // Show success message with share information if available
        if (result.sharesPurchased && result.sharePrice && result.potentialPayout) {
          addToast({
            type: 'success',
            title: 'Bet Placed Successfully!',
            message: `Shares: ${result.sharesPurchased.toFixed(4)} | Price: $${result.sharePrice.toFixed(4)} | Potential Payout: $${result.potentialPayout.toFixed(2)}`
          })
        } else {
          addToast({
            type: 'success',
            title: 'Bet Placed Successfully!',
            message: `Your ${outcome.toUpperCase()} bet has been placed.`
          })
        }

        setAmount('')
        setSuccess(true)
        setTimeout(() => setSuccess(false), 3000)
      } else if (effectiveBetType === 'seed') {
        const seedAmount = parseFloat(seededAmount)
        if (seedAmount <= 0) {
          throw new Error('Seeded amount must be greater than 0')
        }

        // Check if market is still active
        if (market.status !== 'active') {
          throw new Error('This market is no longer accepting bets')
        }

        // Use BettingEngine to seed bet
        const result = await BettingEngine.seedBet(
          user.id,
          market.id,
          outcome,
          seedAmount
        )

        if (!result.success) {
          throw new Error(result.error)
        }

        addToast({
          type: 'success',
          title: 'Bet Seeded Successfully!',
          message: `Your ${outcome.toUpperCase()} bet has been seeded. Make it live when ready.`
        })

        setSeededAmount('')
        setSuccess(true)
        setTimeout(() => setSuccess(false), 3000)
      }

      onBetPlaced()
    } catch (err: unknown) {
      const errorMessage = err instanceof Error ? err.message : 'An error occurred'
      setError(errorMessage)
      addToast({
        type: 'error',
        title: 'Bet Failed',
        message: errorMessage
      })
    } finally {
      setLoading(false)
    }
  }

  if (market.status !== 'active') {
    return (
      <div className="bg-gray-100 p-6 rounded-lg">
        <p className="text-black text-center">
          This market is {market.status} and no longer accepting bets.
        </p>
      </div>
    )
  }

  return (
    <div className={`bg-white p-6 rounded-lg border relative transition-all duration-300 ${
      success ? 'border-green-300 bg-green-50' : 'border-gray-200'
    }`}>
      {loading && <LoadingSpinner overlay text="Processing bet..." />}
      
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-lg font-semibold">Place Your Bet</h3>
        {success && (
          <div className="flex items-center text-green-600">
            <svg className="w-5 h-5 mr-2" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
            </svg>
            <span className="text-sm font-medium">Success!</span>
          </div>
        )}
      </div>
      
      <form onSubmit={handleSubmit} className="space-y-4">
        {/* Outcome Selection */}
        <div className="grid grid-cols-2 gap-4">
          <button
            type="button"
            onClick={() => setOutcome('yes')}
            className={`p-4 rounded-lg border-2 transition-colors ${
              outcome === 'yes'
                ? 'border-green-500 bg-green-50 text-green-700'
                : 'border-gray-200 hover:border-green-300'
            }`}
          >
            <div className="text-center">
              <div className="text-2xl font-bold text-green-600">
                {((market.yes_price || 0.5) * 100).toFixed(1)}%
              </div>
              <div className="text-sm font-medium">YES</div>
            </div>
          </button>
          
          <button
            type="button"
            onClick={() => setOutcome('no')}
            className={`p-4 rounded-lg border-2 transition-colors ${
              outcome === 'no'
                ? 'border-red-500 bg-red-50 text-red-700'
                : 'border-gray-200 hover:border-red-300'
            }`}
          >
            <div className="text-center">
              <div className="text-2xl font-bold text-red-600">
                {((market.no_price || 0.5) * 100).toFixed(1)}%
              </div>
              <div className="text-sm font-medium">NO</div>
            </div>
          </button>
        </div>

        {/* Betting Type Selection */}
        <div>
          <label className="block text-sm font-medium text-black mb-2">
            Betting Type
          </label>
          <div className="grid grid-cols-2 gap-4">
            <button
              type="button"
              onClick={() => setBetType('live')}
              className={`p-3 rounded-lg border-2 transition-colors text-center ${
                effectiveBetType === 'live'
                  ? 'border-blue-500 bg-blue-50 text-blue-700'
                  : 'border-gray-200 hover:border-blue-300'
              }`}
            >
              <div className="font-medium">Live Bet</div>
              <div className="text-xs text-black">Immediate trading</div>
            </button>
            
            <button
              type="button"
              onClick={() => setBetType('seed')}
              disabled={(market.live_volume || 0) > 0}
              className={`p-3 rounded-lg border-2 transition-colors text-center ${
                (market.live_volume || 0) > 0
                  ? 'border-gray-200 bg-gray-100 text-gray-400 cursor-not-allowed'
                  : effectiveBetType === 'seed'
                  ? 'border-purple-500 bg-purple-50 text-purple-700'
                  : 'border-gray-200 hover:border-purple-300'
              }`}
            >
              <div className="font-medium">Seed Bet</div>
              <div className="text-xs text-black">
                {(market.live_volume || 0) > 0 ? 'Market is live' : 'Pre-fund for later'}
              </div>
            </button>
          </div>
          <p className="mt-2 text-xs text-black">
            {effectiveBetType === 'live' 
              ? 'Your bet will be matched immediately if there are opposing bets.'
              : (market.live_volume || 0) > 0
              ? 'Seeding is only available before live trading begins.'
              : 'Your funds will be locked until you make the bet live later. Seeded bets get better payout ratios when they win.'
            }
          </p>
        </div>

        {/* Amount Input */}
        {effectiveBetType === 'live' ? (
          <div>
            <label htmlFor="amount" className="block text-sm font-medium text-black mb-2">
              Amount to Bet ($)
            </label>
            <input
              id="amount"
              type="number"
              step="0.01"
              min="0.01"
              value={amount}
              onChange={(e) => setAmount(e.target.value)}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              style={{ color: 'black' }}
              placeholder="0.00"
              required
            />
          </div>
        ) : (
          <div>
            <label htmlFor="seededAmount" className="block text-sm font-medium text-black mb-2">
              Amount to Seed ($)
            </label>
            <input
              id="seededAmount"
              type="number"
              step="0.01"
              min="0.01"
              value={seededAmount}
              onChange={(e) => setSeededAmount(e.target.value)}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-purple-500 focus:border-transparent"
              style={{ color: 'black' }}
              placeholder="0.00"
              required
            />
            <p className="mt-2 text-sm text-black">
              Seeded funds will be locked until you make the bet live
            </p>
          </div>
        )}

        {/* Bet Summary */}
        {(amount || seededAmount) && (
          <div className="bg-gray-50 p-4 rounded-lg">
            <div className="flex justify-between text-sm">
              <span className="text-black">Current Price:</span>
              <span className="font-medium text-black">{(currentPrice * 100).toFixed(1)}%</span>
            </div>
            {betType === 'live' && amount && (
              <>
                <div className="flex justify-between text-sm">
                  <span className="text-black">Shares:</span>
                  <span className="font-medium text-black">{shares}</span>
                </div>
                <div className="flex justify-between text-sm font-semibold border-t pt-2 mt-2">
                  <span className="text-black">Total Cost:</span>
                  <span className="font-medium text-black">${amount}</span>
                </div>
              </>
            )}
            {betType === 'seed' && seededAmount && (
              <div className="flex justify-between text-sm font-semibold border-t pt-2 mt-2">
                <span className="text-black">Amount to Seed:</span>
                <span className="font-medium text-black">${seededAmount}</span>
              </div>
            )}
          </div>
        )}

        {/* Error Message */}
        {error && (
          <div className="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
            {error}
          </div>
        )}

        {/* Submit Button */}
        <button
          type="submit"
          disabled={loading || (betType === 'live' ? !amount || parseFloat(amount) <= 0 : !seededAmount || parseFloat(seededAmount) <= 0)}
          className={`w-full py-3 px-4 rounded-lg font-medium transition-colors ${
            betType === 'live' 
              ? (outcome === 'yes' ? 'bg-green-600 hover:bg-green-700 text-white' : 'bg-red-600 hover:bg-red-700 text-white')
              : 'bg-purple-600 hover:bg-purple-700 text-white'
          } disabled:opacity-50 disabled:cursor-not-allowed`}
        >
          {loading ? (
            <ButtonLoadingSpinner text={betType === 'live' ? 'Placing Bet...' : 'Seeding Bet...'} />
          ) : (
            betType === 'live' ? `Bet ${outcome.toUpperCase()}` : `Seed ${outcome.toUpperCase()}`
          )}
        </button>
      </form>
    </div>
  )
}
