'use client'

import { useState } from 'react'
import Link from 'next/link'
import { Market } from '@/lib/types'

interface MarketCardProps {
  market: Market
}

export default function MarketCard({ market }: MarketCardProps) {
  const [isHovered, setIsHovered] = useState(false)

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active':
        return 'bg-green-100 text-green-800'
      case 'closed':
        return 'bg-yellow-100 text-yellow-800'
      case 'resolved':
        return 'bg-blue-100 text-blue-800'
      default:
        return 'bg-gray-100 text-black'
    }
  }

  const getOutcomeColor = (outcome: string | null) => {
    switch (outcome) {
      case 'yes':
        return 'bg-green-500 text-white'
      case 'no':
        return 'bg-red-500 text-white'
      default:
        return 'bg-gray-500 text-white'
    }
  }

  const formatPrice = (price: number) => {
    // Ensure the price is between 0 and 1, then convert to percentage
    const normalizedPrice = Math.max(0, Math.min(1, price || 0.5))
    return (normalizedPrice * 100).toFixed(1)
  }

  const formatVolume = (volume: number) => {
    if (volume >= 1000000) {
      return `$${(volume / 1000000).toFixed(1)}M`
    } else if (volume >= 1000) {
      return `$${(volume / 1000).toFixed(1)}K`
    } else {
      return `$${volume.toFixed(0)}`
    }
  }

  const timeRemaining = () => {
    const now = new Date()
    const endDate = new Date(market.end_date)
    const diff = endDate.getTime() - now.getTime()
    
    if (diff <= 0) return 'Ended'
    
    const days = Math.floor(diff / (1000 * 60 * 60 * 24))
    const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60))
    
    if (days > 0) return `${days}d ${hours}h`
    return `${hours}h`
  }

  return (
    <div
      className={`bg-white rounded-lg shadow-md border border-gray-200 transition-all duration-200 h-full flex flex-col ${
        isHovered ? 'shadow-lg transform -translate-y-1' : ''
      }`}
      onMouseEnter={() => setIsHovered(true)}
      onMouseLeave={() => setIsHovered(false)}
    >
      <div className="p-6 flex flex-col h-full">
        {/* Header */}
        <div className="flex items-start justify-between mb-4">
          <div className="flex-1">
            <div className="flex items-center gap-2 mb-2">
              <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(market.status)}`}>
                {market.status.toUpperCase()}
              </span>
              {market.outcome && (
                <span className={`px-2 py-1 text-xs font-medium rounded-full ${getOutcomeColor(market.outcome)}`}>
                  {market.outcome.toUpperCase()}
                </span>
              )}
            </div>
            <h3 className="text-lg font-semibold text-black mb-2 line-clamp-2">
              {market.title}
            </h3>
            <p className="text-black text-sm line-clamp-2">
              {market.description}
            </p>
          </div>
        </div>

        {/* Market Stats */}
        <div className="grid grid-cols-2 gap-4 mb-4">
          <div className="text-center">
            <div className="text-2xl font-bold text-green-600">
              {formatPrice(market.yes_price || 0.5)}
            </div>
            <div className="text-xs text-black uppercase tracking-wide">
              YES
            </div>
          </div>
          <div className="text-center">
            <div className="text-2xl font-bold text-red-600">
              {formatPrice(market.no_price || 0.5)}
            </div>
            <div className="text-xs text-black uppercase tracking-wide">
              NO
            </div>
          </div>
        </div>

        {/* Volume and Time */}
        <div className="flex justify-between items-center text-sm text-black mb-2">
          <span>Total Pool: {formatVolume(market.total_volume)}</span>
          <span>{timeRemaining()}</span>
        </div>
        
        {/* Pool breakdown */}
        <div className="text-xs text-black mb-2">
          <div className="flex justify-between">
            <span>YES Pool: {formatVolume(market.yes_pool || 0)}</span>
            <span>NO Pool: {formatVolume(market.no_pool || 0)}</span>
          </div>
        </div>
        
        {/* Seeded vs Live breakdown */}
        <div className="text-xs text-black mb-4">
          <div className="flex justify-between">
            <span>Seeded: {formatVolume(market.seeded_volume || 0)}</span>
            <span>Live: {formatVolume(market.live_volume || 0)}</span>
          </div>
          {market.creator_earnings && market.creator_earnings > 0 && (
            <div className="text-center mt-1 text-blue-600">
              Creator: {formatVolume(market.creator_earnings)}
            </div>
          )}
        </div>

        {/* Category */}
        <div className="flex items-center justify-between mt-auto">
          <span className="text-xs text-black uppercase tracking-wide">
            {market.category}
          </span>
          <Link
            href={`/markets/${market.id}`}
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors text-sm font-medium"
          >
            View Market
          </Link>
        </div>
      </div>
    </div>
  )
}
