'use client'

import { useEffect, useState } from 'react'
import { supabase } from '@/lib/supabase-client'

interface MarketChartProps {
  marketId: string
}

interface PriceData {
  timestamp: string
  yes_price: number
  no_price: number
}

export default function MarketChart({ marketId }: MarketChartProps) {
  const [priceData, setPriceData] = useState<PriceData[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    const fetchPriceHistory = async () => {
      try {
        setLoading(true)
        
        // Try to fetch real price history
        const { data: priceHistory, error: historyError } = await supabase
          .from('price_history')
          .select('timestamp, yes_price, no_price')
          .eq('market_id', marketId)
          .gte('timestamp', new Date(Date.now() - 7 * 24 * 60 * 60 * 1000).toISOString()) // Last 7 days
          .order('timestamp', { ascending: true })

        if (historyError) {
          console.warn('Price history not available:', historyError)
          setPriceData([])
        } else if (priceHistory && priceHistory.length > 0) {
          // Use real price history data
          const formattedData: PriceData[] = priceHistory.map(point => ({
            timestamp: point.timestamp,
            yes_price: parseFloat(point.yes_price.toString()),
            no_price: parseFloat(point.no_price.toString())
          }))
          
          setPriceData(formattedData)
        } else {
          setPriceData([])
        }
      } catch (error) {
        console.error('Error fetching price history:', error)
        setPriceData([])
      } finally {
        setLoading(false)
      }
    }

    fetchPriceHistory()
  }, [marketId])

  if (loading) {
    return (
      <div className="h-64 flex items-center justify-center">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    )
  }

  const formatDate = (timestamp: string) => {
    return new Date(timestamp).toLocaleDateString('en-US', { 
      month: 'short', 
      day: 'numeric' 
    })
  }

  return (
    <div className="h-64">
      <div className="flex items-center justify-between mb-4">
        <div className="flex items-center gap-4">
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 bg-green-500 rounded-full"></div>
            <span className="text-sm text-black">YES</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 bg-red-500 rounded-full"></div>
            <span className="text-sm text-black">NO</span>
          </div>
        </div>
      </div>
      
      <div className="relative h-48 bg-gray-50 rounded-lg p-4">
        <svg className="w-full h-full" viewBox="0 0 400 180">
          {/* Grid lines */}
          <defs>
            <pattern id="grid" width="40" height="18" patternUnits="userSpaceOnUse">
              <path d="M 40 0 L 0 0 0 18" fill="none" stroke="#e5e7eb" strokeWidth="0.5"/>
            </pattern>
          </defs>
          <rect width="100%" height="100%" fill="url(#grid)" />
          
          {/* Price lines */}
          {priceData.length > 1 && (
            <>
              {/* YES price line */}
              <polyline
                fill="none"
                stroke="#10b981"
                strokeWidth="2"
                points={priceData.map((point, index) => 
                  `${(index / (priceData.length - 1)) * 380 + 10},${(1 - point.yes_price) * 160 + 10}`
                ).join(' ')}
              />
              
              {/* NO price line */}
              <polyline
                fill="none"
                stroke="#ef4444"
                strokeWidth="2"
                points={priceData.map((point, index) => 
                  `${(index / (priceData.length - 1)) * 380 + 10},${(1 - point.no_price) * 160 + 10}`
                ).join(' ')}
              />
            </>
          )}
          
          {/* Current price indicators */}
          {priceData.length > 0 && (
            <>
              <circle
                cx={380}
                cy={(1 - priceData[priceData.length - 1].yes_price) * 160 + 10}
                r="4"
                fill="#10b981"
              />
              <circle
                cx={380}
                cy={(1 - priceData[priceData.length - 1].no_price) * 160 + 10}
                r="4"
                fill="#ef4444"
              />
            </>
          )}
        </svg>
        
        {/* Y-axis labels */}
        <div className="absolute left-2 top-2 text-xs text-black">100%</div>
        <div className="absolute left-2 top-1/2 transform -translate-y-1/2 text-xs text-black">50%</div>
        <div className="absolute left-2 bottom-2 text-xs text-black">0%</div>
        
        {/* X-axis labels */}
        <div className="absolute bottom-0 left-0 right-0 flex justify-between px-4 pb-2">
          <span className="text-xs text-black">{formatDate(priceData[0]?.timestamp || '')}</span>
          <span className="text-xs text-black">Today</span>
        </div>
      </div>
    </div>
  )
}
