'use client'

import { Market } from '@/lib/types'

interface MarketPriceDisplayProps {
  market: Market
  className?: string
}

export default function MarketPriceDisplay({ market, className = '' }: MarketPriceDisplayProps) {
  const formatPrice = (price: number) => {
    return `${(price * 100).toFixed(1)}%`
  }

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD',
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(amount)
  }

  return (
    <div className={`bg-gray-50 p-3 rounded-lg border border-gray-200 ${className}`}>
      <div className="flex items-center justify-between mb-2">
        <span className="text-xs font-medium text-gray-700">Live Market Prices</span>
        <div className="flex items-center space-x-1">
          <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
          <span className="text-xs text-gray-500">Live</span>
        </div>
      </div>
      
      {market.market_type === 'multi_outcome' && market.outcomes ? (
        <div className="space-y-2">
          {market.outcomes.slice(0, 3).map((outcome) => (
            <div key={outcome.id} className="flex items-center justify-between text-sm">
              <div className="flex items-center space-x-2">
                <div 
                  className="w-4 h-4 rounded-full flex items-center justify-center text-xs"
                  style={{ backgroundColor: (outcome.color || '#3B82F6') + '20' }}
                >
                  {outcome.icon || '👤'}
                </div>
                <span className="font-medium text-gray-900">{outcome.name}</span>
              </div>
              <div className="text-right">
                <div 
                  className="font-bold"
                  style={{ color: outcome.color || '#3B82F6' }}
                >
                  {formatPrice(outcome.price)}
                </div>
                <div className="text-xs text-gray-500">
                  ${outcome.pool.toFixed(0)} pool
                </div>
              </div>
            </div>
          ))}
          {market.outcomes.length > 3 && (
            <div className="text-xs text-gray-500 text-center">
              +{market.outcomes.length - 3} more outcomes
            </div>
          )}
        </div>
      ) : (
        <div className="grid grid-cols-2 gap-3 text-sm">
          <div className="text-center">
            <div className="font-bold text-green-600">
              {formatPrice(market.yes_price || 0.5)}
            </div>
            <div className="text-xs text-gray-500">YES</div>
            <div className="text-xs text-gray-500">
              ${(market.yes_pool || 0).toFixed(0)} pool
            </div>
          </div>
          <div className="text-center">
            <div className="font-bold text-red-600">
              {formatPrice(market.no_price || 0.5)}
            </div>
            <div className="text-xs text-gray-500">NO</div>
            <div className="text-xs text-gray-500">
              ${(market.no_pool || 0).toFixed(0)} pool
            </div>
          </div>
        </div>
      )}
      
      <div className="mt-2 pt-2 border-t border-gray-200">
        <div className="flex justify-between text-xs text-gray-500">
          <span>Total Volume:</span>
          <span>{formatCurrency(market.total_volume || 0)}</span>
        </div>
        <div className="flex justify-between text-xs text-gray-500">
          <span>Live Volume:</span>
          <span>{formatCurrency(market.live_volume || 0)}</span>
        </div>
      </div>
    </div>
  )
}
