'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'
import { useToast } from '@/components/ui/Toast'

interface MarketOutcome {
  id: string
  name: string
  description?: string
  color: string
  icon: string
  share_price: number
  pool_amount: number
  shares_sold: number
  shares_remaining: number
  sort_order: number
}

interface Market {
  id: string
  title: string
  description: string
  status: string
  market_type: string
  total_shares: number
}

interface MultiOutcomeBetFormProps {
  market: Market
  onBetPlaced: () => void
}

export default function MultiOutcomeBetForm({ market, onBetPlaced }: MultiOutcomeBetFormProps) {
  const [outcomes, setOutcomes] = useState<MarketOutcome[]>([])
  const [selectedOutcome, setSelectedOutcome] = useState<string>('')
  const [amount, setAmount] = useState('')
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState('')
  const [outcomesLoading, setOutcomesLoading] = useState(true)
  const [success, setSuccess] = useState(false)
  const { addToast } = useToast()

  useEffect(() => {
    fetchOutcomes()
  }, [market.id])

  const fetchOutcomes = async () => {
    try {
      setOutcomesLoading(true)
      const { data, error } = await supabase
        .from('market_outcomes')
        .select('*')
        .eq('market_id', market.id)
        .order('sort_order', { ascending: true })

      if (error) {
        console.error('Error fetching outcomes:', error)
        setError('Failed to load market outcomes')
        return
      }

      setOutcomes(data || [])
      if (data && data.length > 0 && !selectedOutcome) {
        setSelectedOutcome(data[0].name)
      }
    } catch (err) {
      console.error('Error in fetchOutcomes:', err)
      setError('Failed to load market outcomes')
    } finally {
      setOutcomesLoading(false)
    }
  }

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD',
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(amount)
  }

  const formatSharePrice = (price: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD',
      minimumFractionDigits: 4,
      maximumFractionDigits: 4,
    }).format(price)
  }

  const formatShares = (shares: number) => {
    return new Intl.NumberFormat('en-US', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 4,
    }).format(shares)
  }

  const calculateSharesFromAmount = (amount: number, price: number) => {
    return amount / price
  }

  const calculateAmountFromShares = (shares: number, price: number) => {
    return shares * price
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)
    setError('')

    try {
      const { data: { user } } = await supabase.auth.getUser()
      if (!user) {
        throw new Error('You must be logged in to place a bet')
      }

      const betAmount = parseFloat(amount)
      if (betAmount <= 0) {
        throw new Error('Amount must be greater than 0')
      }

      if (!selectedOutcome) {
        throw new Error('Please select an outcome')
      }

      // Check if market is still active
      if (market.status !== 'active') {
        throw new Error('This market is no longer accepting bets')
      }

      // Use the multi-outcome betting function
      const { data: result, error: betError } = await supabase
        .rpc('place_multi_outcome_bet', {
          p_user_id: user.id,
          p_market_id: market.id,
          p_outcome_name: selectedOutcome,
          p_amount: betAmount,
          p_shares: null
        })
        .single()

      if (betError) {
        throw new Error(betError.message)
      }

      const betResult = result as any
      if (!betResult.success) {
        throw new Error(betResult.message)
      }

      // Show success message with share information
      if (betResult.shares_purchased && betResult.share_price && betResult.potential_payout) {
        addToast({
          type: 'success',
          title: 'Bet Placed Successfully!',
          message: `Outcome: ${selectedOutcome} | Shares: ${betResult.shares_purchased.toFixed(4)} | Price: ${formatSharePrice(betResult.share_price)} | Potential Payout: ${formatCurrency(betResult.potential_payout)}`
        })
      } else {
        addToast({
          type: 'success',
          title: 'Bet Placed Successfully!',
          message: `Your bet on ${selectedOutcome} has been placed.`
        })
      }

      setAmount('')
      setSuccess(true)
      setTimeout(() => setSuccess(false), 3000)
      onBetPlaced()
      fetchOutcomes() // Refresh outcomes to show updated prices
    } catch (err: unknown) {
      const errorMessage = err instanceof Error ? err.message : 'An error occurred'
      setError(errorMessage)
      addToast({
        type: 'error',
        title: 'Bet Failed',
        message: errorMessage
      })
    } finally {
      setLoading(false)
    }
  }

  if (market.status !== 'active') {
    return (
      <div className="bg-gray-100 p-6 rounded-lg">
        <p className="text-black text-center">
          This market is {market.status} and no longer accepting bets.
        </p>
      </div>
    )
  }

  if (outcomesLoading) {
    return (
      <div className="bg-white p-6 rounded-lg border border-gray-200">
        <h3 className="text-lg font-semibold mb-4">Loading Market Outcomes...</h3>
        <div className="animate-pulse space-y-4">
          <div className="h-4 bg-gray-200 rounded w-3/4"></div>
          <div className="h-4 bg-gray-200 rounded w-1/2"></div>
          <div className="h-4 bg-gray-200 rounded w-2/3"></div>
        </div>
      </div>
    )
  }

  if (outcomes.length === 0) {
    return (
      <div className="bg-white p-6 rounded-lg border border-gray-200">
        <h3 className="text-lg font-semibold mb-4">No Outcomes Available</h3>
        <p className="text-black text-center">
          This market doesn't have any outcomes set up yet.
        </p>
      </div>
    )
  }

  const selectedOutcomeData = outcomes.find(o => o.name === selectedOutcome)
  const estimatedShares = selectedOutcomeData && amount ? 
    calculateSharesFromAmount(parseFloat(amount), selectedOutcomeData.share_price) : 0

  return (
    <div className="bg-white p-6 rounded-lg border border-gray-200 relative">
      {loading && <LoadingSpinner overlay text="Processing bet..." />}
      
      <h3 className="text-lg font-semibold mb-4">Place Your Bet</h3>
      
      {error && (
        <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg">
          <p className="text-red-700 text-sm">{error}</p>
        </div>
      )}
      
      <form onSubmit={handleSubmit} className="space-y-6">
        {/* Outcome Selection */}
        <div>
          <label className="block text-sm font-medium text-black mb-3">
            Select Outcome
          </label>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
            {outcomes.map((outcome) => (
              <button
                key={outcome.id}
                type="button"
                onClick={() => setSelectedOutcome(outcome.name)}
                className={`p-4 rounded-lg border-2 transition-colors text-left ${
                  selectedOutcome === outcome.name
                    ? 'border-blue-500 bg-blue-50'
                    : 'border-gray-200 hover:border-blue-300'
                }`}
              >
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-3">
                    <span className="text-2xl">{outcome.icon}</span>
                    <div>
                      <div className="font-medium text-black">{outcome.name}</div>
                      {outcome.description && (
                        <div className="text-sm text-gray-600">{outcome.description}</div>
                      )}
                    </div>
                  </div>
                  <div className="text-right">
                    <div className="text-lg font-bold" style={{ color: outcome.color }}>
                      {formatSharePrice(outcome.share_price)}
                    </div>
                    <div className="text-xs text-gray-600">per share</div>
                  </div>
                </div>
                
                {/* Outcome Stats */}
                <div className="mt-3 pt-3 border-t border-gray-200">
                  <div className="grid grid-cols-2 gap-4 text-xs">
                    <div>
                      <span className="text-gray-600">Pool:</span>
                      <div className="font-medium">{formatCurrency(outcome.pool_amount)}</div>
                    </div>
                    <div>
                      <span className="text-gray-600">Shares:</span>
                      <div className="font-medium">{formatShares(outcome.shares_sold)} sold</div>
                    </div>
                  </div>
                </div>
              </button>
            ))}
          </div>
        </div>

        {/* Bet Amount */}
        <div>
          <label htmlFor="amount" className="block text-sm font-medium text-black mb-2">
            Bet Amount (USD)
          </label>
          <input
            type="number"
            id="amount"
            value={amount}
            onChange={(e) => setAmount(e.target.value)}
            step="0.01"
            min="0.01"
            placeholder="10.00"
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors !text-black"
            style={{ color: 'black' }}
            required
          />
          
          {/* Estimated Shares */}
          {selectedOutcomeData && amount && parseFloat(amount) > 0 && (
            <div className="mt-2 p-3 bg-blue-50 border border-blue-200 rounded-lg">
              <div className="flex items-center justify-between text-sm">
                <span className="text-blue-800 font-medium">Estimated Shares:</span>
                <span className="text-blue-900 font-bold">{estimatedShares.toFixed(4)}</span>
              </div>
              <div className="flex items-center justify-between text-sm mt-1">
                <span className="text-blue-800 font-medium">Potential Payout:</span>
                <span className="text-blue-900 font-bold">{formatCurrency(estimatedShares * 1.00)}</span>
              </div>
              <p className="mt-1 text-xs text-blue-700">
                Each share is worth $1.00 if this outcome wins
              </p>
            </div>
          )}
        </div>

        {/* Submit Button */}
        <button
          type="submit"
          disabled={loading || !selectedOutcome || !amount || parseFloat(amount) <= 0}
          className="w-full bg-blue-600 text-white py-3 px-4 rounded-lg font-medium hover:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
        >
          {loading ? 'Placing Bet...' : `Place Bet on ${selectedOutcome}`}
        </button>

        {/* Market Info */}
        <div className="bg-gray-50 p-4 rounded-lg">
          <h4 className="font-medium text-black mb-2">How Multi-Outcome Betting Works</h4>
          <ul className="text-sm text-gray-700 space-y-1">
            <li>• Buy shares in any outcome you think will win</li>
            <li>• Each share costs the current market price</li>
            <li>• If your outcome wins, each share pays $1.00</li>
            <li>• Share prices change based on demand and remaining supply</li>
            <li>• Only one outcome can win per market</li>
          </ul>
        </div>
      </form>
    </div>
  )
}
