'use client'

import { useState } from 'react'
import { Plus, X, GripVertical } from 'lucide-react'

interface OutcomeOption {
  id: string
  name: string
  description: string
  color: string
  icon: string
}

interface MultiOutcomeCreatorProps {
  outcomes: OutcomeOption[]
  onChange: (outcomes: OutcomeOption[]) => void
  maxOutcomes?: number
}

const PRESET_COLORS = [
  '#3B82F6', '#EF4444', '#10B981', '#F59E0B', '#8B5CF6',
  '#EC4899', '#06B6D4', '#84CC16', '#F97316', '#6366F1'
]

const PRESET_ICONS = [
  '👤', '🏛️', '🎯', '⭐', '🔥', '💎', '🚀', '🏆', '🎪', '🌟'
]

export default function MultiOutcomeCreator({ 
  outcomes, 
  onChange, 
  maxOutcomes = 10 
}: MultiOutcomeCreatorProps) {
  const [draggedIndex, setDraggedIndex] = useState<number | null>(null)

  const addOutcome = () => {
    if (outcomes.length >= maxOutcomes) return

    const newOutcome: OutcomeOption = {
      id: `outcome-${Date.now()}`,
      name: '',
      description: '',
      color: PRESET_COLORS[outcomes.length % PRESET_COLORS.length],
      icon: PRESET_ICONS[outcomes.length % PRESET_ICONS.length]
    }

    onChange([...outcomes, newOutcome])
  }

  const removeOutcome = (id: string) => {
    onChange(outcomes.filter(outcome => outcome.id !== id))
  }

  const updateOutcome = (id: string, field: keyof OutcomeOption, value: string) => {
    onChange(outcomes.map(outcome => 
      outcome.id === id ? { ...outcome, [field]: value } : outcome
    ))
  }

  const moveOutcome = (fromIndex: number, toIndex: number) => {
    const newOutcomes = [...outcomes]
    const [movedOutcome] = newOutcomes.splice(fromIndex, 1)
    newOutcomes.splice(toIndex, 0, movedOutcome)
    onChange(newOutcomes)
  }

  const handleDragStart = (e: React.DragEvent, index: number) => {
    setDraggedIndex(index)
    e.dataTransfer.effectAllowed = 'move'
  }

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault()
    e.dataTransfer.dropEffect = 'move'
  }

  const handleDrop = (e: React.DragEvent, dropIndex: number) => {
    e.preventDefault()
    if (draggedIndex !== null && draggedIndex !== dropIndex) {
      moveOutcome(draggedIndex, dropIndex)
    }
    setDraggedIndex(null)
  }

  return (
    <div className="space-y-4">
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-semibold text-gray-900">
          Market Outcomes ({outcomes.length}/{maxOutcomes})
        </h3>
        <button
          type="button"
          onClick={addOutcome}
          disabled={outcomes.length >= maxOutcomes}
          className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
        >
          <Plus className="w-4 h-4" />
          Add Outcome
        </button>
      </div>

      {outcomes.length === 0 && (
        <div className="text-center py-8 text-gray-500">
          <p>No outcomes added yet. Click "Add Outcome" to get started.</p>
          <p className="text-sm mt-1">Example: For "Who will be president?", add candidates like "John Doe", "Jane Smith", etc.</p>
        </div>
      )}

      <div className="space-y-3">
        {outcomes.map((outcome, index) => (
          <div
            key={outcome.id}
            draggable
            onDragStart={(e) => handleDragStart(e, index)}
            onDragOver={handleDragOver}
            onDrop={(e) => handleDrop(e, index)}
            className="bg-white border border-gray-200 rounded-lg p-4 hover:shadow-sm transition-shadow"
          >
            <div className="flex items-start gap-3">
              {/* Drag Handle */}
              <div className="flex-shrink-0 mt-2 cursor-move text-gray-400 hover:text-gray-600">
                <GripVertical className="w-4 h-4" />
              </div>

              {/* Outcome Number */}
              <div className="flex-shrink-0 w-8 h-8 bg-gray-100 rounded-full flex items-center justify-center text-sm font-medium text-gray-600">
                {index + 1}
              </div>

              {/* Outcome Content */}
              <div className="flex-1 space-y-3">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                  {/* Name */}
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Outcome Name *
                    </label>
                    <input
                      type="text"
                      value={outcome.name}
                      onChange={(e) => updateOutcome(outcome.id, 'name', e.target.value)}
                      placeholder="e.g., John Doe, Jane Smith"
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      required
                    />
                  </div>

                  {/* Icon */}
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Icon
                    </label>
                    <div className="flex items-center gap-2">
                      <input
                        type="text"
                        value={outcome.icon}
                        onChange={(e) => updateOutcome(outcome.id, 'icon', e.target.value)}
                        placeholder="👤"
                        className="w-16 px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 text-center"
                        maxLength={2}
                      />
                      <div className="flex gap-1">
                        {PRESET_ICONS.slice(0, 5).map((icon) => (
                          <button
                            key={icon}
                            type="button"
                            onClick={() => updateOutcome(outcome.id, 'icon', icon)}
                            className={`w-8 h-8 rounded border-2 flex items-center justify-center text-sm ${
                              outcome.icon === icon 
                                ? 'border-blue-500 bg-blue-50' 
                                : 'border-gray-200 hover:border-gray-300'
                            }`}
                          >
                            {icon}
                          </button>
                        ))}
                      </div>
                    </div>
                  </div>
                </div>

                {/* Description */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Description (Optional)
                  </label>
                  <input
                    type="text"
                    value={outcome.description}
                    onChange={(e) => updateOutcome(outcome.id, 'description', e.target.value)}
                    placeholder="Brief description of this outcome"
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  />
                </div>

                {/* Color Picker */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Color
                  </label>
                  <div className="flex gap-2 flex-wrap">
                    {PRESET_COLORS.map((color) => (
                      <button
                        key={color}
                        type="button"
                        onClick={() => updateOutcome(outcome.id, 'color', color)}
                        className={`w-8 h-8 rounded-full border-2 ${
                          outcome.color === color 
                            ? 'border-gray-400' 
                            : 'border-gray-200 hover:border-gray-300'
                        }`}
                        style={{ backgroundColor: color }}
                        title={color}
                      />
                    ))}
                  </div>
                </div>
              </div>

              {/* Remove Button */}
              <button
                type="button"
                onClick={() => removeOutcome(outcome.id)}
                className="flex-shrink-0 p-2 text-gray-400 hover:text-red-600 transition-colors"
                disabled={outcomes.length <= 2}
              >
                <X className="w-4 h-4" />
              </button>
            </div>
          </div>
        ))}
      </div>

      {outcomes.length > 0 && (
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
          <h4 className="font-medium text-blue-900 mb-2">Preview</h4>
          <div className="flex flex-wrap gap-2">
            {outcomes.map((outcome, index) => (
              <div
                key={outcome.id}
                className="flex items-center gap-2 px-3 py-2 rounded-lg text-sm"
                style={{ backgroundColor: outcome.color + '20', color: outcome.color }}
              >
                <span>{outcome.icon}</span>
                <span className="font-medium">
                  {outcome.name || `Outcome ${index + 1}`}
                </span>
              </div>
            ))}
          </div>
        </div>
      )}

      {outcomes.length >= maxOutcomes && (
        <div className="text-center py-4 text-gray-500">
          <p>Maximum of {maxOutcomes} outcomes reached.</p>
        </div>
      )}
    </div>
  )
}
