'use client'

import { useState } from 'react'
import { ChevronDown, ChevronUp } from 'lucide-react'
import { MarketOutcome } from '@/lib/types'

interface MultiOutcomeDisplayProps {
  outcomes: MarketOutcome[]
  maxVisible?: number
  showAll?: boolean
  compact?: boolean
}

export default function MultiOutcomeDisplay({ 
  outcomes, 
  maxVisible = 3, 
  showAll = false,
  compact = false 
}: MultiOutcomeDisplayProps) {
  const [expanded, setExpanded] = useState(false)
  
  if (!outcomes || outcomes.length === 0) {
    return (
      <div className="text-center py-2 text-gray-500 text-sm">
        No outcomes available
      </div>
    )
  }

  const visibleOutcomes = showAll || expanded ? outcomes : outcomes.slice(0, maxVisible)
  const hasMore = outcomes.length > maxVisible && !showAll

  const formatPrice = (price: number) => {
    return `${(price * 100).toFixed(1)}%`
  }

  if (compact) {
    return (
      <div className="space-y-1">
        {visibleOutcomes.map((outcome, index) => (
          <div key={outcome.id} className="flex items-center justify-between">
            <div className="flex items-center gap-2 min-w-0 flex-1">
              <div 
                className="w-6 h-6 rounded-full flex items-center justify-center text-xs flex-shrink-0"
                style={{ backgroundColor: (outcome.color || '#3B82F6') + '20' }}
              >
                {outcome.icon || '👤'}
              </div>
              <span className="text-xs text-black truncate">{outcome.name}</span>
            </div>
            <span 
              className="text-sm font-bold ml-2"
              style={{ color: outcome.color || '#3B82F6' }}
            >
              {formatPrice(outcome.price)}
            </span>
          </div>
        ))}
        
        {hasMore && (
          <button
            onClick={() => setExpanded(!expanded)}
            className="flex items-center gap-1 text-xs text-blue-600 hover:text-blue-800 w-full justify-center py-1"
          >
            {expanded ? (
              <>
                <ChevronUp className="w-3 h-3" />
                Show Less
              </>
            ) : (
              <>
                <ChevronDown className="w-3 h-3" />
                +{outcomes.length - maxVisible} More
              </>
            )}
          </button>
        )}
      </div>
    )
  }

  return (
    <div className="space-y-2">
      {/* Circular outcome display */}
      <div className="flex flex-wrap gap-2 mb-3">
        {visibleOutcomes.map((outcome, index) => (
          <div
            key={outcome.id}
            className="flex flex-col items-center group cursor-pointer"
            title={`${outcome.name} - ${formatPrice(outcome.price)}`}
          >
            {/* Circular outcome indicator */}
            <div 
              className="w-12 h-12 rounded-full flex items-center justify-center text-lg font-bold text-white shadow-md hover:shadow-lg transition-shadow"
              style={{ backgroundColor: outcome.color || '#3B82F6' }}
            >
              {outcome.icon || '👤'}
            </div>
            
            {/* Outcome name */}
            <div className="text-xs text-center mt-1 max-w-16 truncate">
              {outcome.name}
            </div>
            
            {/* Price */}
            <div 
              className="text-xs font-bold"
              style={{ color: outcome.color || '#3B82F6' }}
            >
              {formatPrice(outcome.price)}
            </div>
          </div>
        ))}
        
        {/* Show more indicator */}
        {hasMore && (
          <div className="flex flex-col items-center">
            <button
              onClick={() => setExpanded(!expanded)}
              className="w-12 h-12 rounded-full bg-gray-100 hover:bg-gray-200 flex items-center justify-center text-gray-600 hover:text-gray-800 transition-colors"
            >
              {expanded ? (
                <ChevronUp className="w-4 h-4" />
              ) : (
                <ChevronDown className="w-4 h-4" />
              )}
            </button>
            <div className="text-xs text-gray-500 mt-1">
              {expanded ? 'Less' : `+${outcomes.length - maxVisible}`}
            </div>
          </div>
        )}
      </div>

      {/* Detailed list view (when expanded) */}
      {expanded && (
        <div className="space-y-1 pt-2 border-t border-gray-200">
          {outcomes.map((outcome) => (
            <div key={outcome.id} className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <div 
                  className="w-4 h-4 rounded-full flex items-center justify-center text-xs"
                  style={{ backgroundColor: (outcome.color || '#3B82F6') + '20' }}
                >
                  {outcome.icon || '👤'}
                </div>
                <span className="text-sm text-black">{outcome.name}</span>
                {outcome.description && (
                  <span className="text-xs text-gray-500">- {outcome.description}</span>
                )}
              </div>
              <div className="flex items-center gap-2">
                <span 
                  className="text-sm font-bold"
                  style={{ color: outcome.color || '#3B82F6' }}
                >
                  {formatPrice(outcome.price)}
                </span>
                <div className="text-xs text-gray-500">
                  ${outcome.pool.toFixed(0)} pool
                </div>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  )
}
