'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'

interface PriceData {
  timestamp: string
  yes_price: number
  no_price: number
}

interface PriceChartProps {
  marketId: string
  className?: string
}

export default function PriceChart({ marketId, className = '' }: PriceChartProps) {
  const [priceData, setPriceData] = useState<PriceData[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    const fetchPriceHistory = async (isBackground = false) => {
      try {
        // Only show loading on initial fetch, not background updates
        if (!isBackground) {
          setLoading(true)
          setError(null)
        }

        // Fetch current market data to use as baseline
        const { data: marketData, error: marketError } = await supabase
          .from('markets')
          .select('yes_price, no_price, total_volume, yes_pool, no_pool')
          .eq('id', marketId)
          .single()

        if (marketError || !marketData) {
          throw new Error('Market not found')
        }

        // Try to fetch real price history first
        const { data: priceHistory, error: historyError } = await supabase
          .from('price_history')
          .select('timestamp, yes_price, no_price, yes_pool, no_pool, total_volume')
          .eq('market_id', marketId)
          .gte('timestamp', new Date(Date.now() - 24 * 60 * 60 * 1000).toISOString()) // Last 24 hours
          .order('timestamp', { ascending: true })

        if (historyError) {
          console.warn('Price history table not available, falling back to current market data:', historyError)
          
          // Fallback: Create minimal price history with current data
          const now = new Date()
          const currentYesPrice = marketData.yes_price || 0.5
          const currentNoPrice = marketData.no_price || 0.5
          
          // Create 5 data points over the last hour
          const fallbackData: PriceData[] = []
          for (let i = 4; i >= 0; i--) {
            const timestamp = new Date(now.getTime() - i * 15 * 60000) // Every 15 minutes
            const variation = (Math.random() - 0.5) * 0.02 // Small random variation
            
            let yesPrice = currentYesPrice + variation
            let noPrice = currentNoPrice - variation
            
            // Normalize prices
            yesPrice = Math.max(0.1, Math.min(0.9, yesPrice))
            noPrice = Math.max(0.1, Math.min(0.9, noPrice))
            
            const total = yesPrice + noPrice
            yesPrice = yesPrice / total
            noPrice = noPrice / total
            
            fallbackData.push({
              timestamp: timestamp.toISOString(),
              yes_price: yesPrice,
              no_price: noPrice
            })
          }
          
          setPriceData(fallbackData)
        } else if (priceHistory && priceHistory.length > 0) {
          // Use real price history data
          const formattedData: PriceData[] = priceHistory.map(point => ({
            timestamp: point.timestamp,
            yes_price: parseFloat(point.yes_price.toString()),
            no_price: parseFloat(point.no_price.toString())
          }))
          
          setPriceData(formattedData)
        } else {
          // No price history available yet - create initial data point with current market data
          const now = new Date()
          const currentYesPrice = marketData.yes_price || 0.5
          const currentNoPrice = marketData.no_price || 0.5
          
          setPriceData([{
            timestamp: now.toISOString(),
            yes_price: currentYesPrice,
            no_price: currentNoPrice
          }])
        }
      } catch (err) {
        if (!isBackground) {
          setError('Failed to fetch price data')
        }
        console.error('Error fetching price history:', err)
      } finally {
        // Only set loading to false on initial fetch
        if (!isBackground) {
          setLoading(false)
        }
      }
    }

    // Initial fetch (with loading state)
    fetchPriceHistory(false)

    // Update price data every 30 seconds (background updates)
    const interval = setInterval(() => {
      fetchPriceHistory(true)
    }, 30000)

    return () => clearInterval(interval)
  }, [marketId])

  if (loading) {
    return (
      <div className={`bg-white rounded-lg p-4 ${className}`}>
        <div className="animate-pulse">
          <div className="h-4 bg-gray-200 rounded w-1/4 mb-4"></div>
          <div className="h-32 bg-gray-200 rounded"></div>
        </div>
      </div>
    )
  }

  if (error) {
    return (
      <div className={`bg-white rounded-lg p-4 ${className}`}>
        <h3 className="text-lg font-semibold text-black mb-2">Price Chart</h3>
        <p className="text-red-600">{error}</p>
      </div>
    )
  }

  if (priceData.length === 0) {
    return (
      <div className={`bg-white rounded-lg p-4 ${className}`}>
        <h3 className="text-lg font-semibold text-black mb-2">Price Chart</h3>
        <p className="text-gray-600">No price data available</p>
      </div>
    )
  }

  // Calculate chart dimensions and scaling
  const chartWidth = 400
  const chartHeight = 200
  const padding = 20
  const maxPrice = 1
  const minPrice = 0

  const scaleY = (price: number) => {
    return chartHeight - padding - ((price - minPrice) / (maxPrice - minPrice)) * (chartHeight - 2 * padding)
  }

  const scaleX = (index: number) => {
    if (priceData.length <= 1) return chartWidth / 2
    return padding + (index / (priceData.length - 1)) * (chartWidth - 2 * padding)
  }

  // Generate SVG paths for YES and NO prices
  const yesPath = priceData
    .map((point, index) => `${index === 0 ? 'M' : 'L'} ${scaleX(index)} ${scaleY(point.yes_price)}`)
    .join(' ')

  const noPath = priceData
    .map((point, index) => `${index === 0 ? 'M' : 'L'} ${scaleX(index)} ${scaleY(point.no_price)}`)
    .join(' ')

  const currentYes = priceData[priceData.length - 1]?.yes_price || 0
  const currentNo = priceData[priceData.length - 1]?.no_price || 0

  return (
    <div className={`bg-white rounded-lg p-4 ${className}`}>
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-lg font-semibold text-black">Price Movement</h3>
        <div className="flex items-center space-x-4 text-sm">
          <div className="flex items-center space-x-2">
            <div className="w-3 h-3 bg-green-500 rounded-full"></div>
            <span className="text-black">YES: {Math.round(currentYes * 100)}%</span>
          </div>
          <div className="flex items-center space-x-2">
            <div className="w-3 h-3 bg-red-500 rounded-full"></div>
            <span className="text-black">NO: {Math.round(currentNo * 100)}%</span>
          </div>
        </div>
      </div>

      <div className="relative">
        <svg width={chartWidth} height={chartHeight} className="border border-gray-200 rounded">
          {/* Grid lines */}
          <defs>
            <pattern id="grid" width="40" height="20" patternUnits="userSpaceOnUse">
              <path d="M 40 0 L 0 0 0 20" fill="none" stroke="#f3f4f6" strokeWidth="1"/>
            </pattern>
          </defs>
          <rect width="100%" height="100%" fill="url(#grid)" />

          {/* Y-axis labels */}
          <text x={5} y={padding + 5} className="text-xs fill-gray-500">100%</text>
          <text x={5} y={chartHeight / 2} className="text-xs fill-gray-500">50%</text>
          <text x={5} y={chartHeight - padding + 5} className="text-xs fill-gray-500">0%</text>

          {/* Price lines */}
          <path
            d={yesPath}
            fill="none"
            stroke="#10b981"
            strokeWidth="2"
            className="drop-shadow-sm"
          />
          <path
            d={noPath}
            fill="none"
            stroke="#ef4444"
            strokeWidth="2"
            className="drop-shadow-sm"
          />

          {/* Current price indicators */}
          <circle
            cx={scaleX(priceData.length - 1)}
            cy={scaleY(currentYes)}
            r="4"
            fill="#10b981"
            className="drop-shadow-sm"
          />
          <circle
            cx={scaleX(priceData.length - 1)}
            cy={scaleY(currentNo)}
            r="4"
            fill="#ef4444"
            className="drop-shadow-sm"
          />

          {/* Time labels */}
          {priceData.length > 0 && (
            <>
              {priceData.length === 1 ? (
                <text x={chartWidth / 2} y={chartHeight - 5} className="text-xs fill-gray-500 text-center">
                  {new Date(priceData[0].timestamp).toLocaleTimeString('en-US', { 
                    hour: 'numeric', 
                    minute: '2-digit',
                    hour12: true 
                  })}
                </text>
              ) : (
                <>
                  <text x={padding} y={chartHeight - 5} className="text-xs fill-gray-500">
                    {new Date(priceData[0].timestamp).toLocaleTimeString('en-US', { 
                      hour: 'numeric', 
                      minute: '2-digit',
                      hour12: true 
                    })}
                  </text>
                  <text x={chartWidth - padding} y={chartHeight - 5} className="text-xs fill-gray-500">
                    {new Date(priceData[priceData.length - 1].timestamp).toLocaleTimeString('en-US', { 
                      hour: 'numeric', 
                      minute: '2-digit',
                      hour12: true 
                    })}
                  </text>
                </>
              )}
            </>
          )}
        </svg>

        {/* Price change indicators */}
        <div className="absolute top-2 right-2 bg-gray-50 rounded p-2 text-xs">
          <div className="flex items-center space-x-2">
            <span className="text-gray-600">Change (1h):</span>
            {priceData.length > 1 ? (
              <span className={`font-medium ${
                currentYes > priceData[0].yes_price ? 'text-green-600' : 'text-red-600'
              }`}>
                YES {currentYes > priceData[0].yes_price ? '+' : ''}
                {Math.round(((currentYes - priceData[0].yes_price) / priceData[0].yes_price) * 100)}%
              </span>
            ) : (
              <span className="text-gray-600">No change data</span>
            )}
          </div>
        </div>
      </div>

      {/* Price statistics */}
      <div className="mt-4 grid grid-cols-2 gap-4 text-sm">
        <div className="bg-gray-50 rounded p-3">
          <div className="text-gray-600 mb-1">YES Pool</div>
          <div className="font-semibold text-black">{Math.round(currentYes * 100)}%</div>
        </div>
        <div className="bg-gray-50 rounded p-3">
          <div className="text-gray-600 mb-1">NO Pool</div>
          <div className="font-semibold text-black">{Math.round(currentNo * 100)}%</div>
        </div>
      </div>
    </div>
  )
}
