'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'
import { useLiveMarketUpdates } from '@/lib/hooks/useLiveMarketUpdates'

interface ShareInfoProps {
  marketId: string
  className?: string
}

interface ShareData {
  total_shares: number
  shares_sold: number
  current_yes_share_price: number
  current_no_share_price: number
  initial_share_price: number
  yes_pool: number
  no_pool: number
  total_volume: number
  // Multi-outcome specific data
  market_type: 'binary' | 'multi_outcome'
  outcomes?: Array<{
    name: string
    pool: number
    color?: string
    icon?: string
  }>
  current_outcome_share_prices?: Record<string, number>
}

export default function ShareInfo({ marketId, className = '' }: ShareInfoProps) {
  const { marketData, loading, error, lastUpdated, isLive } = useLiveMarketUpdates({
    marketId,
    enabled: true,
    refreshInterval: 5000 // 5 seconds
  })

  // Convert marketData to ShareData format
  const shareData: ShareData | null = marketData ? {
    total_shares: marketData.total_shares || 1000,
    shares_sold: marketData.shares_sold || 0,
    // Calculate share prices from market prices with safety checks
    current_yes_share_price: (() => {
      const price = marketData.yes_price
      if (isNaN(price) || price <= 0) return 0.50
      return Math.max(0.50, Math.min(1.00, price))
    })(),
    current_no_share_price: (() => {
      const price = marketData.no_price
      if (isNaN(price) || price <= 0) return 0.50
      return Math.max(0.50, Math.min(1.00, price))
    })(),
    initial_share_price: 0.50, // Default initial price
    yes_pool: marketData.yes_pool || 0,
    no_pool: marketData.no_pool || 0,
    total_volume: marketData.total_volume || 0
  } : null

  // Debug logging
  useEffect(() => {
    if (marketData) {
      console.log('ShareInfo marketData:', {
        yes_price: marketData.yes_price,
        no_price: marketData.no_price,
        yes_pool: marketData.yes_pool,
        no_pool: marketData.no_pool,
        total_volume: marketData.total_volume,
        calculated_yes_share: shareData?.current_yes_share_price,
        calculated_no_share: shareData?.current_no_share_price
      })
    }
  }, [marketData, shareData])

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD',
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(amount)
  }

  const formatSharePrice = (price: number) => {
    // Safety check for NaN or invalid values
    if (isNaN(price) || price <= 0) {
      return '$0.50'
    }
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD',
      minimumFractionDigits: 4,
      maximumFractionDigits: 4,
    }).format(price)
  }

  const formatPercentage = (value: number) => {
    return `${(value * 100).toFixed(1)}%`
  }

  if (loading) {
    return (
      <div className={`bg-white rounded-xl shadow-sm border border-gray-200 p-6 ${className}`}>
        <h2 className="text-lg font-semibold text-black mb-4">Share Information</h2>
        <div className="animate-pulse space-y-4">
          <div className="h-4 bg-gray-200 rounded w-3/4"></div>
          <div className="h-4 bg-gray-200 rounded w-1/2"></div>
          <div className="h-4 bg-gray-200 rounded w-2/3"></div>
        </div>
      </div>
    )
  }

  if (error || !shareData) {
    return (
      <div className={`bg-white rounded-xl shadow-sm border border-gray-200 p-6 ${className}`}>
        <h2 className="text-lg font-semibold text-black mb-4">Share Information</h2>
        <div className="text-center py-8">
          <div className="text-red-600 mb-2">⚠️</div>
          <p className="text-black">{error || 'No share data available'}</p>
        </div>
      </div>
    )
  }

  const sharesRemaining = shareData.total_shares - shareData.shares_sold
  
  const sharesSoldPercentage = shareData.total_shares > 0 
    ? (shareData.shares_sold / shareData.total_shares) * 100 
    : 0

  const sharesRemainingPercentage = shareData.total_shares > 0 
    ? (sharesRemaining / shareData.total_shares) * 100 
    : 0

  return (
    <div className={`bg-white rounded-xl shadow-sm border border-gray-200 p-6 ${className}`}>
      <h2 className="text-lg font-semibold text-black mb-6">Share Information</h2>
      
      {/* Share Distribution */}
      <div className="mb-6">
        <h3 className="text-md font-medium text-black mb-3">Share Distribution</h3>
        <div className="space-y-3">
          <div className="flex justify-between items-center">
            <span className="text-black">Total Shares:</span>
            <span className="font-medium text-black">{shareData.total_shares.toLocaleString()}</span>
          </div>
          <div className="flex justify-between items-center">
            <span className="text-black">Shares Sold:</span>
            <span className="font-medium text-green-600">{shareData.shares_sold.toLocaleString()} ({sharesSoldPercentage.toFixed(1)}%)</span>
          </div>
          <div className="flex justify-between items-center">
            <span className="text-black">Shares Remaining:</span>
            <span className="font-medium text-blue-600">{sharesRemaining.toLocaleString()} ({sharesRemainingPercentage.toFixed(1)}%)</span>
          </div>
        </div>
        
        {/* Progress Bar */}
        <div className="mt-4">
          <div className="w-full bg-gray-200 rounded-full h-2">
            <div 
              className="bg-green-500 h-2 rounded-full transition-all duration-300" 
              style={{ width: `${sharesSoldPercentage}%` }}
            ></div>
          </div>
          <div className="flex justify-between text-xs text-black mt-1">
            <span>Sold</span>
            <span>Remaining</span>
          </div>
        </div>
      </div>

      {/* Share Prices */}
      <div className="mb-6">
        <h3 className="text-md font-medium text-black mb-3">Current Share Prices</h3>
        {marketData?.market_type === 'multi_outcome' ? (
          // Multi-outcome market - show individual outcome prices
          <div className="space-y-3">
            {marketData.outcomes?.map((outcome, index) => {
              const sharePrice = (() => {
                const price = outcome.price
                if (isNaN(price) || price <= 0) return 0.50
                return Math.max(0.50, Math.min(1.00, price))
              })()
              
              return (
                <div 
                  key={outcome.id} 
                  className="border rounded-lg p-3"
                  style={{ 
                    borderColor: outcome.color ? `${outcome.color}40` : '#E5E7EB',
                    backgroundColor: outcome.color ? `${outcome.color}10` : '#F9FAFB'
                  }}
                >
                  <div className="flex items-center space-x-2 mb-2">
                    {outcome.icon && <span className="text-lg">{outcome.icon}</span>}
                    <div className="text-sm font-medium" style={{ color: outcome.color || '#374151' }}>
                      {outcome.name}
                    </div>
                  </div>
                  <div className="text-lg font-bold" style={{ color: outcome.color || '#374151' }}>
                    {formatSharePrice(sharePrice)}
                  </div>
                  <div className="text-xs text-gray-600">
                    Pool: {formatCurrency(outcome.pool || 0)}
                  </div>
                </div>
              )
            })}
          </div>
        ) : (
          // Binary market - show YES/NO prices
          <div className="grid grid-cols-2 gap-4">
            <div className="bg-green-50 border border-green-200 rounded-lg p-3">
              <div className="text-green-800 text-sm font-medium">YES Shares</div>
              <div className="text-green-900 text-lg font-bold">{formatSharePrice(shareData.current_yes_share_price)}</div>
              <div className="text-green-700 text-xs">Pool: {formatCurrency(shareData.yes_pool)}</div>
            </div>
            <div className="bg-red-50 border border-red-200 rounded-lg p-3">
              <div className="text-red-800 text-sm font-medium">NO Shares</div>
              <div className="text-red-900 text-lg font-bold">{formatSharePrice(shareData.current_no_share_price)}</div>
              <div className="text-red-700 text-xs">Pool: {formatCurrency(shareData.no_pool)}</div>
            </div>
          </div>
        )}
      </div>

      {/* Initial Pricing */}
      <div className="mb-6">
        <h3 className="text-md font-medium text-black mb-3">Initial Pricing</h3>
        {marketData?.market_type === 'multi_outcome' ? (
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-3">
            <div className="text-blue-800 text-sm font-medium">Multi-Outcome Market</div>
            <div className="text-blue-900 text-lg font-bold">$0.50 per share</div>
            <div className="text-blue-700 text-xs mt-1">
              All outcomes start at equal probability (33.3% each)
            </div>
          </div>
        ) : (
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-3">
            <div className="text-blue-800 text-sm font-medium">Initial Share Price</div>
            <div className="text-blue-900 text-lg font-bold">{formatSharePrice(shareData.initial_share_price)}</div>
            <div className="text-blue-700 text-xs mt-1">
              Based on seeding amount ÷ total shares
            </div>
          </div>
        )}
      </div>

      {/* Market Summary */}
      <div className="border-t border-gray-200 pt-4">
        <h3 className="text-md font-medium text-black mb-3">Market Summary</h3>
        <div className="grid grid-cols-2 gap-4 text-sm">
          <div className="flex justify-between">
            <span className="text-black">Total Volume:</span>
            <span className="font-medium text-black">{formatCurrency(shareData.total_volume)}</span>
          </div>
          <div className="flex justify-between">
            <span className="text-black">Total Pool:</span>
            <span className="font-medium text-black">
              {marketData?.market_type === 'multi_outcome' 
                ? formatCurrency(marketData.outcomes?.reduce((sum, outcome) => sum + (outcome.pool || 0), 0) || 0)
                : formatCurrency(shareData.yes_pool + shareData.no_pool)
              }
            </span>
          </div>
        </div>
      </div>

      {/* Live Data Indicator */}
      <div className="flex items-center justify-between mt-4 pt-4 border-t border-gray-200">
        <div className="flex items-center space-x-2">
          <div className={`w-2 h-2 rounded-full ${isLive ? 'bg-green-500 animate-pulse' : 'bg-yellow-500'}`}></div>
          <span className="text-xs text-black">
            {isLive ? 'Live data' : 'Updating...'}
          </span>
        </div>
        {lastUpdated && (
          <span className="text-xs text-gray-500">
            {lastUpdated.toLocaleTimeString()}
          </span>
        )}
      </div>
    </div>
  )
}
