'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'
import { Market, MarketOutcome } from '@/lib/types'
import { useToast } from '@/components/ui/Toast'
import { LoadingSpinner } from '@/components/ui/LoadingSpinner'

interface UnifiedBetFormProps {
  market: Market
  onBetPlaced: () => void
}

export default function UnifiedBetForm({ market, onBetPlaced }: UnifiedBetFormProps) {
  const [outcomes, setOutcomes] = useState<MarketOutcome[]>([])
  const [selectedOutcome, setSelectedOutcome] = useState<string>('')
  const [amount, setAmount] = useState('')
  const [seededAmount, setSeededAmount] = useState('')
  const [betType, setBetType] = useState<'live' | 'seed'>('live')
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState('')
  const [success, setSuccess] = useState(false)
  const [outcomesLoading, setOutcomesLoading] = useState(false)
  const [sharesExhausted, setSharesExhausted] = useState<boolean>(false)
  const { addToast } = useToast()

  // Fetch outcomes for multi-outcome markets
  useEffect(() => {
    if (market.market_type === 'multi_outcome') {
      fetchOutcomes()
    }
  }, [market.id, market.market_type])

  // Update local outcomes when market data changes (live updates)
  useEffect(() => {
    if (market.market_type === 'multi_outcome' && market.outcomes) {
      setOutcomes(market.outcomes)
    }
  }, [market.outcomes, market.market_type])

  // Check for share exhaustion
  useEffect(() => {
    const checkShareExhaustion = async () => {
      try {
        const { BettingEngine } = await import('@/lib/betting')
        const exhaustionCheck = await BettingEngine.checkShareExhaustion(market.id)
        setSharesExhausted(exhaustionCheck.exhausted)
      } catch (error) {
        console.error('Error checking share exhaustion:', error)
      }
    }

    checkShareExhaustion()
  }, [market.id, market.status, market.shares_sold, market.total_shares])

  const fetchOutcomes = async () => {
    try {
      setOutcomesLoading(true)
      const { data, error } = await supabase
        .from('market_outcomes')
        .select('*')
        .eq('market_id', market.id)
        .order('sort_order', { ascending: true })

      if (error) {
        console.error('Error fetching outcomes:', error)
        setError('Failed to load market outcomes')
        return
      }

      setOutcomes(data || [])
      if (data && data.length > 0 && !selectedOutcome) {
        setSelectedOutcome(data[0].name)
      }
    } catch (err) {
      console.error('Error in fetchOutcomes:', err)
      setError('Failed to load market outcomes')
    } finally {
      setOutcomesLoading(false)
    }
  }

  // Automatically switch to live betting if market is already live and seeding is selected
  const isMarketLive = (market.live_volume || 0) > 0
  const effectiveBetType = isMarketLive && betType === 'seed' ? 'live' : betType

  // Calculate current price and shares - reactive to market updates
  const getCurrentPrice = () => {
    if (market.market_type === 'multi_outcome') {
      // Use market.outcomes if available (from live updates), otherwise use local outcomes
      const marketOutcomes = market.outcomes || outcomes
      const outcome = marketOutcomes.find(o => o.name === selectedOutcome)
      return outcome?.price || 0.5
    } else {
      return selectedOutcome === 'yes' ? (market.yes_price || 0.5) : (market.no_price || 0.5)
    }
  }

  // Calculate current share price for the selected outcome
  const getCurrentSharePrice = () => {
    if (market.market_type === 'multi_outcome') {
      // Use market.outcomes if available (from live updates), otherwise use local outcomes
      const marketOutcomes = market.outcomes || outcomes
      const outcome = marketOutcomes.find(o => o.name === selectedOutcome)
      
      // For multi-outcome markets, convert probability to share price
      if (outcome) {
        const probability = outcome.price || 0.5
        // Convert probability (0-1) to share price ($0.50 - $1.00)
        return Math.max(0.50, Math.min(1.00, probability))
      }
      
      // Final fallback
      return 0.50
    } else {
      // For binary markets, convert probability to share price
      if (selectedOutcome === 'yes') {
        const probability = market.yes_price || 0.5
        return Math.max(0.50, Math.min(1.00, probability))
      } else {
        const probability = market.no_price || 0.5
        return Math.max(0.50, Math.min(1.00, probability))
      }
    }
  }

  const currentPrice = getCurrentPrice()
  const currentSharePrice = getCurrentSharePrice()
  
  // Safety check to prevent NaN values
  const safeSharePrice = isNaN(currentSharePrice) || currentSharePrice <= 0 ? 0.50 : currentSharePrice
  const shares = amount ? (parseFloat(amount) / safeSharePrice).toFixed(2) : '0.00'

  // Debug log to track price updates
  useEffect(() => {
    if (amount && selectedOutcome) {
      console.log('Share calculation updated:', {
        selectedOutcome,
        currentPrice,
        currentSharePrice,
        safeSharePrice,
        shares,
        marketType: market.market_type,
        hasMarketOutcomes: !!market.outcomes,
        hasLocalOutcomes: outcomes.length > 0,
        // Debug share price calculation
        rawOutcomePrice: market.market_type === 'multi_outcome' ? 
          (market.outcomes || outcomes).find(o => o.name === selectedOutcome)?.price : null,
        rawMarketPrice: market.market_type === 'binary' ? 
          (selectedOutcome === 'yes' ? market.yes_price : market.no_price) : null
      })
    }
  }, [selectedOutcome, currentPrice, currentSharePrice, shares, amount, market.market_type, market.outcomes, outcomes.length])

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD',
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(amount)
  }

  const formatPrice = (price: number) => {
    return `${(price * 100).toFixed(1)}%`
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)
    setError('')
    setSuccess(false)

    try {
      const { data: { user } } = await supabase.auth.getUser()
      if (!user) {
        throw new Error('You must be logged in to place a bet')
      }

      const betAmount = parseFloat(amount)
      if (betAmount <= 0) {
        throw new Error('Amount must be greater than 0')
      }

      if (!selectedOutcome) {
        throw new Error('Please select an outcome')
      }

      // Check if market is still active
      if (market.status !== 'active') {
        throw new Error('This market is no longer accepting bets')
      }

      let result

      if (market.market_type === 'multi_outcome') {
        // Use multi-outcome betting function
        console.log('Placing multi-outcome bet:', {
          user_id: user.id,
          market_id: market.id,
          outcome_name: selectedOutcome,
          amount: betAmount
        })

        try {
          // Find the outcome ID for the selected outcome
          const marketOutcomes = market.outcomes || outcomes
          const selectedOutcomeData = marketOutcomes.find(o => o.name === selectedOutcome)
          
          if (!selectedOutcomeData) {
            throw new Error('Selected outcome not found')
          }

          // Get user balance first
          const { data: profile, error: profileError } = await supabase
            .from('profiles')
            .select('balance')
            .eq('id', user.id)
            .single()

          if (profileError || !profile) {
            throw new Error('User not found')
          }

          if (profile.balance < betAmount) {
            throw new Error('Insufficient balance')
          }

          // Calculate shares based on outcome price
          const outcomePrice = selectedOutcomeData.price || 0.5
          const shares = betAmount / outcomePrice

          // Create bet record directly
          const { data: bet, error: betError } = await supabase
            .from('bets')
            .insert({
              user_id: user.id,
              market_id: market.id,
              outcome: selectedOutcome,
              amount: betAmount,
              price: outcomePrice,
              shares: shares,
              status: 'matched'
            })
            .select()
            .single()

          if (betError) {
            console.error('Bet creation error:', betError)
            throw new Error('Failed to create bet record: ' + betError.message)
          }

          // Update user balance
          const { error: balanceError } = await supabase
            .from('profiles')
            .update({ balance: profile.balance - betAmount })
            .eq('id', user.id)

          if (balanceError) {
            console.error('Balance update error:', balanceError)
            // Note: Bet is already created, so we don't rollback here
          }

          // Format the result to match expected structure
          result = {
            success: true,
            bet_id: bet.id,
            shares_purchased: shares,
            share_price: outcomePrice,
            potential_payout: shares
          }
        } catch (error: any) {
          console.error('Multi-outcome bet exception:', error)
          throw error
        }
      } else {
        // Use binary betting function
        console.log('Placing binary bet:', {
          user_id: user.id,
          market_id: market.id,
          outcome: selectedOutcome,
          amount: betAmount
        })

        try {
          // Get user balance first
          const { data: profile, error: profileError } = await supabase
            .from('profiles')
            .select('balance')
            .eq('id', user.id)
            .single()

          if (profileError || !profile) {
            throw new Error('User not found')
          }

          if (profile.balance < betAmount) {
            throw new Error('Insufficient balance')
          }

          // Calculate shares based on market price
          const marketPrice = selectedOutcome === 'yes' ? (market.yes_price || 0.5) : (market.no_price || 0.5)
          const shares = betAmount / marketPrice

          // Create bet record directly
          const { data: bet, error: betError } = await supabase
            .from('bets')
            .insert({
              user_id: user.id,
              market_id: market.id,
              outcome: selectedOutcome,
              amount: betAmount,
              price: marketPrice,
              shares: shares,
              status: 'matched'
            })
            .select()
            .single()

          if (betError) {
            console.error('Bet creation error:', betError)
            throw new Error('Failed to create bet record: ' + betError.message)
          }

          // Update user balance
          const { error: balanceError } = await supabase
            .from('profiles')
            .update({ balance: profile.balance - betAmount })
            .eq('id', user.id)

          if (balanceError) {
            console.error('Balance update error:', balanceError)
            // Note: Bet is already created, so we don't rollback here
          }

          // Format the result to match expected structure
          result = {
            success: true,
            bet_id: bet.id,
            shares_purchased: shares,
            share_price: marketPrice,
            potential_payout: shares
          }
        } catch (error: any) {
          console.error('Binary bet exception:', error)
          throw error
        }
      }

      const betResult = result as any
      if (!betResult.success) {
        throw new Error(betResult.message)
      }

      // Show success message with share information
      if (betResult.shares_purchased && betResult.share_price && betResult.potential_payout) {
        const sharesPurchased = typeof betResult.shares_purchased === 'number' 
          ? betResult.shares_purchased.toFixed(4) 
          : '0.0000'
        const sharePrice = typeof betResult.share_price === 'number' 
          ? formatCurrency(betResult.share_price) 
          : '$0.00'
        const potentialPayout = typeof betResult.potential_payout === 'number' 
          ? formatCurrency(betResult.potential_payout) 
          : '$0.00'
        
        addToast({
          type: 'success',
          title: 'Bet Placed Successfully!',
          message: `Outcome: ${selectedOutcome} | Shares: ${sharesPurchased} | Price: ${sharePrice} | Potential Payout: ${potentialPayout}`
        })
      } else {
        addToast({
          type: 'success',
          title: 'Bet Placed Successfully!',
          message: `Your bet on ${selectedOutcome} has been placed.`
        })
      }

      setSuccess(true)
      setAmount('')
      setSeededAmount('')
      setSelectedOutcome(market.market_type === 'multi_outcome' ? (outcomes[0]?.name || '') : 'yes')
      
      // Call the callback to refresh data
      onBetPlaced()

    } catch (err: any) {
      console.error('Error placing bet:', err)
      setError(err.message || 'Failed to place bet')
      addToast({
        type: 'error',
        title: 'Bet Failed',
        message: err.message || 'Failed to place bet'
      })
    } finally {
      setLoading(false)
    }
  }

  if (outcomesLoading) {
    return (
      <div className="text-center py-8">
        <LoadingSpinner />
        <p className="text-gray-600 mt-2">Loading market outcomes...</p>
      </div>
    )
  }

  return (
    <div className="bg-white rounded-lg border border-gray-200 relative">
      {loading && <LoadingSpinner overlay text="Processing bet..." />}
      
      <div className="p-4 border-b border-gray-200">
        <h3 className="text-lg font-semibold text-gray-900">Place Your Bet</h3>
      </div>
      
      <div className="p-4">
        {error && (
          <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-lg">
            <p className="text-red-700 text-sm">{error}</p>
          </div>
        )}
        
        <form onSubmit={handleSubmit} className="space-y-4">
        {/* Outcome Selection */}
        <div>
          <label className="block text-sm font-medium text-black mb-3">
            Select Outcome
          </label>
          
          {market.market_type === 'multi_outcome' ? (
            // Multi-outcome selection - compact grid
            <div className="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-2 xl:grid-cols-3 gap-2">
              {(market.outcomes || outcomes).map((outcome) => (
                <button
                  key={outcome.id}
                  type="button"
                  onClick={() => setSelectedOutcome(outcome.name)}
                  className={`p-3 rounded-lg border-2 transition-colors text-left ${
                    selectedOutcome === outcome.name
                      ? 'border-blue-500 bg-blue-50'
                      : 'border-gray-200 hover:border-blue-300'
                  }`}
                >
                  <div className="flex items-center space-x-2">
                    <div 
                      className="w-6 h-6 rounded-full flex items-center justify-center text-xs flex-shrink-0"
                      style={{ backgroundColor: (outcome.color || '#3B82F6') + '20' }}
                    >
                      {outcome.icon || '👤'}
                    </div>
                    <div className="min-w-0 flex-1">
                      <div className="font-medium text-black text-sm truncate">{outcome.name}</div>
                      <div 
                        className="text-xs font-bold"
                        style={{ color: outcome.color || '#3B82F6' }}
                      >
                        {formatPrice(outcome.price)}
                      </div>
                    </div>
                  </div>
                </button>
              ))}
            </div>
          ) : (
            // Binary selection - compact layout
            <div className="grid grid-cols-2 gap-3">
              <button
                type="button"
                onClick={() => setSelectedOutcome('yes')}
                className={`p-3 rounded-lg border-2 transition-colors ${
                  selectedOutcome === 'yes'
                    ? 'border-green-500 bg-green-50 text-green-700'
                    : 'border-gray-200 hover:border-green-300'
                }`}
              >
                <div className="text-center">
                  <div className="text-lg font-bold text-green-600">
                    {formatPrice(market.yes_price || 0.5)}
                  </div>
                  <div className="text-sm font-medium">YES</div>
                </div>
              </button>
              
              <button
                type="button"
                onClick={() => setSelectedOutcome('no')}
                className={`p-3 rounded-lg border-2 transition-colors ${
                  selectedOutcome === 'no'
                    ? 'border-red-500 bg-red-50 text-red-700'
                    : 'border-gray-200 hover:border-red-300'
                }`}
              >
                <div className="text-center">
                  <div className="text-lg font-bold text-red-600">
                    {formatPrice(market.no_price || 0.5)}
                  </div>
                  <div className="text-sm font-medium">NO</div>
                </div>
              </button>
            </div>
          )}
        </div>

        {/* Betting Type Selection - Compact */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Betting Type
          </label>
          <div className="grid grid-cols-2 gap-2">
            <button
              type="button"
              onClick={() => setBetType('live')}
              className={`p-2 rounded-lg border-2 transition-colors ${
                effectiveBetType === 'live'
                  ? 'border-blue-500 bg-blue-50 text-blue-700'
                  : 'border-gray-200 hover:border-blue-300'
              }`}
            >
              <div className="text-center">
                <div className="font-medium text-sm">Live</div>
                <div className="text-xs text-gray-600">Trade</div>
              </div>
            </button>
            
            <button
              type="button"
              onClick={() => setBetType('seed')}
              className={`p-2 rounded-lg border-2 transition-colors ${
                effectiveBetType === 'seed'
                  ? 'border-purple-500 bg-purple-50 text-purple-700'
                  : 'border-gray-200 hover:border-purple-300'
              }`}
            >
              <div className="text-center">
                <div className="font-medium text-sm">Seed</div>
                <div className="text-xs text-gray-600">Liquidity</div>
              </div>
            </button>
          </div>
        </div>

        {/* Amount Input */}
        <div>
          <label htmlFor="amount" className="block text-sm font-medium text-gray-700 mb-2">
            Bet Amount ($)
          </label>
          <input
            type="number"
            id="amount"
            value={amount}
            onChange={(e) => setAmount(e.target.value)}
            placeholder="0.00"
            min="0"
            step="0.01"
            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors text-sm"
            required
          />
        </div>

        {/* Share Information - Market Type Specific */}
        {amount && (
          <div className="bg-gray-50 p-3 rounded-lg border border-gray-200">
            <div className="flex items-center justify-between mb-2">
              <span className="text-xs font-medium text-gray-700">Share Information</span>
              <div className="flex items-center space-x-1">
                <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                <span className="text-xs text-gray-500">Live</span>
              </div>
            </div>
            <div className="grid grid-cols-2 gap-3 text-sm">
              <div>
                <span className="text-gray-600 text-xs">Share Price:</span>
                <div className="font-bold text-base text-blue-600">{formatCurrency(safeSharePrice)}</div>
              </div>
              <div>
                <span className="text-gray-600 text-xs">Your Shares:</span>
                <div className="font-bold text-base text-green-600">{shares}</div>
              </div>
            </div>
            <div className="mt-2 p-2 bg-blue-50 rounded border border-blue-200">
              <div className="text-xs text-blue-800">
                {market.market_type === 'multi_outcome' ? (
                  <>
                    If <strong>{selectedOutcome}</strong> wins, you'll receive <strong>{formatCurrency(parseFloat(shares))}</strong> per share
                  </>
                ) : (
                  <>
                    If <strong>{selectedOutcome.toUpperCase()}</strong> wins, you'll receive <strong>{formatCurrency(parseFloat(shares))}</strong> per share
                  </>
                )}
              </div>
            </div>
          </div>
        )}

        {/* Submit Button */}
        <button
          type="submit"
          disabled={loading || !selectedOutcome || !amount}
          className="w-full bg-blue-600 text-white py-2.5 px-4 rounded-lg font-medium hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors text-sm"
        >
          {loading ? 'Placing Bet...' : `Place Bet on ${selectedOutcome}`}
        </button>
        </form>
      </div>
    </div>
  )
}
