'use client'

import { Bet } from '@/lib/types'

interface ActiveBetsProps {
  bets: any[]
}

export default function ActiveBets({ bets }: ActiveBetsProps) {
  const formatCurrency = (amount: number) => {
    return `$${amount.toFixed(2)}`
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    })
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'matched':
        return 'bg-green-100 text-green-800'
      case 'pending':
        return 'bg-yellow-100 text-yellow-800'
      case 'cancelled':
        return 'bg-red-100 text-red-800'
      default:
        return 'bg-gray-100 text-black'
    }
  }

  const getOutcomeColor = (outcome: string) => {
    switch (outcome) {
      case 'yes':
        return 'bg-green-100 text-green-800'
      case 'no':
        return 'bg-red-100 text-red-800'
      default:
        return 'bg-gray-100 text-black'
    }
  }

  const getMarketStatusColor = (status: string) => {
    switch (status) {
      case 'active':
        return 'bg-blue-100 text-blue-800'
      case 'closed':
        return 'bg-yellow-100 text-yellow-800'
      case 'resolved':
        return 'bg-green-100 text-green-800'
      default:
        return 'bg-gray-100 text-black'
    }
  }

  const calculatePotentialReturn = (bet: any) => {
    if (bet.markets?.status === 'resolved') {
      if (bet.markets.outcome === bet.outcome) {
        return formatCurrency(bet.shares) // Full return for winning bet
      } else {
        return '$0.00' // Nothing for losing bet
      }
    }
    return formatCurrency(bet.shares) // Potential return if market resolves in favor
  }

  return (
    <div>
      <div className="flex items-center justify-between mb-6">
        <h3 className="text-lg font-semibold">My Bets</h3>
        <div className="text-sm text-black">
          {bets.length} total bets
        </div>
      </div>

      <div className="overflow-x-auto">
        <table className="min-w-full divide-y divide-gray-200">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Market
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Outcome
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Amount
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Shares
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Price
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Status
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Potential Return
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-black uppercase tracking-wider">
                Date
              </th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {bets.map((bet) => (
              <tr key={bet.id} className="hover:bg-gray-50">
                <td className="px-6 py-4 whitespace-nowrap">
                  <div className="flex items-center">
                    <div>
                      <div className="text-sm font-medium text-black max-w-xs truncate">
                        {bet.markets?.title || 'Unknown Market'}
                      </div>
                      <div className="flex items-center gap-2 mt-1">
                        <span className={`px-2 py-1 text-xs font-medium rounded-full ${getMarketStatusColor(bet.markets?.status || 'unknown')}`}>
                          {bet.markets?.status || 'unknown'}
                        </span>
                        {bet.markets?.outcome && (
                          <span className={`px-2 py-1 text-xs font-medium rounded-full ${
                            bet.markets.outcome === 'yes' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                          }`}>
                            {bet.markets.outcome.toUpperCase()}
                          </span>
                        )}
                      </div>
                    </div>
                  </div>
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${getOutcomeColor(bet.outcome)}`}>
                    {bet.outcome.toUpperCase()}
                  </span>
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                  {formatCurrency(bet.amount)}
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                  {bet.shares.toFixed(2)}
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                  {(bet.price * 100).toFixed(1)}%
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${getStatusColor(bet.status)}`}>
                    {bet.status}
                  </span>
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                  <span className={`font-medium ${
                    bet.markets?.status === 'resolved' && bet.markets.outcome === bet.outcome
                      ? 'text-green-600'
                      : bet.markets?.status === 'resolved' && bet.markets.outcome !== bet.outcome
                      ? 'text-red-600'
                      : 'text-black'
                  }`}>
                    {calculatePotentialReturn(bet)}
                  </span>
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-black">
                  {formatDate(bet.created_at)}
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>

      {bets.length === 0 && (
        <div className="text-center py-12">
          <div className="text-black mb-4">
            <svg className="mx-auto h-12 w-12" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1" />
            </svg>
          </div>
          <h3 className="text-lg font-medium text-black mb-2">No bets found</h3>
          <p className="text-black">Start placing bets to see them here</p>
        </div>
      )}
    </div>
  )
}
