'use client'

import React, { useState, memo } from 'react'
import { Profile, BetWithMarket, TransactionWithMarket } from '@/lib/types'
import PortfolioStats from './PortfolioStats'
import ActiveBets from './ActiveBets'
import TransactionHistory from './TransactionHistory'
import RecentActivityOptimized from '@/components/ui/RecentActivityOptimized'
import { usePerformanceMonitor } from '@/lib/hooks/usePerformanceMonitor'

interface PortfolioDashboardProps {
  profile: Profile | null
  bets: BetWithMarket[]
  transactions: TransactionWithMarket[]
  stats: {
    balance: number
    totalBets: number
    totalVolume: number
    totalWinnings: number
    activeBets: number
    winRate: number
  }
}

const TabButton = memo(({ 
  tab, 
  isActive, 
  onClick 
}: { 
  tab: { id: string; name: string; icon: string }, 
  isActive: boolean, 
  onClick: () => void 
}) => (
  <button
    onClick={onClick}
    className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors ${
      isActive
        ? 'border-blue-500 text-blue-600'
        : 'border-transparent text-black hover:text-black hover:border-gray-300'
    }`}
  >
    <span className="mr-2">{tab.icon}</span>
    {tab.name}
  </button>
))

TabButton.displayName = 'TabButton'

function PortfolioDashboardOptimized({ 
  profile, 
  bets, 
  transactions, 
  stats 
}: PortfolioDashboardProps) {
  const [activeTab, setActiveTab] = useState('overview')
  
  // Monitor performance in development
  usePerformanceMonitor('PortfolioDashboard', process.env.NODE_ENV === 'development')

  const tabs = [
    { id: 'overview', name: 'Overview', icon: '📊' },
    { id: 'bets', name: 'My Bets', icon: '💰' },
    { id: 'transactions', name: 'Transactions', icon: '📋' },
  ]

  return (
    <div className="space-y-6">
      {/* Stats Cards */}
      <PortfolioStats stats={stats} />

      {/* Tabs */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="border-b border-gray-200">
          <nav className="-mb-px flex space-x-8 px-6">
            {tabs.map((tab) => (
              <TabButton
                key={tab.id}
                tab={tab}
                isActive={activeTab === tab.id}
                onClick={() => setActiveTab(tab.id)}
              />
            ))}
          </nav>
        </div>

        <div className="p-6">
          {activeTab === 'overview' && (
            <div className="space-y-6">
              <div>
                <h3 className="text-lg font-semibold mb-4">Recent Activity</h3>
                <RecentActivityOptimized limit={10} showMarketLinks={true} />
              </div>
            </div>
          )}

          {activeTab === 'bets' && (
            <ActiveBets bets={bets} />
          )}

          {activeTab === 'transactions' && (
            <TransactionHistory transactions={transactions} />
          )}
        </div>
      </div>
    </div>
  )
}

export default memo(PortfolioDashboardOptimized)
