import { Stats } from '@/lib/types'

interface PortfolioStatsProps {
  stats: {
    balance: number
    totalBets: number
    totalVolume: number
    totalWinnings: number
    activeBets: number
    winRate: number
  }
}

export default function PortfolioStats({ stats }: PortfolioStatsProps) {
  const formatCurrency = (amount: number) => {
    return `$${amount.toFixed(2)}`
  }

  const statCards = [
    {
      title: 'Current Balance',
      value: formatCurrency(stats.balance),
      icon: '💰',
      color: 'green',
      description: 'Available funds'
    },
    {
      title: 'Total Bets',
      value: stats.totalBets,
      icon: '🎯',
      color: 'blue',
      description: 'All time bets'
    },
    {
      title: 'Total Volume',
      value: formatCurrency(stats.totalVolume),
      icon: '📊',
      color: 'purple',
      description: 'Amount wagered'
    },
    {
      title: 'Total Winnings',
      value: formatCurrency(stats.totalWinnings),
      icon: '🏆',
      color: 'yellow',
      description: 'Profits earned'
    },
    {
      title: 'Active Bets',
      value: stats.activeBets,
      icon: '⏳',
      color: 'orange',
      description: 'Pending resolution'
    },
    {
      title: 'Win Rate',
      value: `${stats.winRate.toFixed(1)}%`,
      icon: '📈',
      color: 'red',
      description: 'Success rate'
    }
  ]

  const getColorClasses = (color: string) => {
    switch (color) {
      case 'green':
        return 'bg-green-500 text-white'
      case 'blue':
        return 'bg-blue-500 text-white'
      case 'purple':
        return 'bg-purple-500 text-white'
      case 'yellow':
        return 'bg-yellow-500 text-white'
      case 'orange':
        return 'bg-orange-500 text-white'
      case 'red':
        return 'bg-red-500 text-white'
      default:
        return 'bg-gray-500 text-white'
    }
  }

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
      {statCards.map((stat, index) => (
        <div key={index} className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center">
            <div className={`w-12 h-12 rounded-lg ${getColorClasses(stat.color)} flex items-center justify-center text-xl`}>
              {stat.icon}
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-black">{stat.title}</p>
              <p className="text-2xl font-bold text-black">{stat.value}</p>
              <p className="text-xs text-black">{stat.description}</p>
            </div>
          </div>
        </div>
      ))}
    </div>
  )
}
