'use client'

import { Transaction } from '@/lib/types'

interface TransactionHistoryProps {
  transactions: any[]
}

export default function TransactionHistory({ transactions }: TransactionHistoryProps) {
  const formatCurrency = (amount: number) => {
    return `$${amount.toFixed(2)}`
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    })
  }

  const getTypeColor = (type: string) => {
    switch (type) {
      case 'deposit':
        return 'bg-green-100 text-green-800'
      case 'withdrawal':
        return 'bg-red-100 text-red-800'
      case 'bet':
        return 'bg-blue-100 text-blue-800'
      case 'win':
        return 'bg-green-100 text-green-800'
      case 'refund':
        return 'bg-yellow-100 text-yellow-800'
      default:
        return 'bg-gray-100 text-black'
    }
  }

  const getAmountColor = (amount: number, type: string) => {
    if (amount > 0) return 'text-green-600'
    if (amount < 0) return 'text-red-600'
    return 'text-black'
  }

  const getTypeIcon = (type: string) => {
    switch (type) {
      case 'deposit':
        return '💰'
      case 'withdrawal':
        return '💸'
      case 'bet':
        return '🎯'
      case 'win':
        return '🏆'
      case 'refund':
        return '↩️'
      default:
        return '📋'
    }
  }

  return (
    <div>
      <div className="flex items-center justify-between mb-6">
        <h3 className="text-lg font-semibold">Transaction History</h3>
        <div className="text-sm text-black">
          {transactions.length} transactions
        </div>
      </div>

      <div className="space-y-4">
        {transactions.map((transaction) => (
          <div key={transaction.id} className="bg-white border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-4">
                <div className="text-2xl">
                  {getTypeIcon(transaction.type)}
                </div>
                <div>
                  <div className="flex items-center space-x-2">
                    <h4 className="text-sm font-medium text-black">
                      {transaction.description}
                    </h4>
                    <span className={`px-2 py-1 text-xs font-medium rounded-full ${getTypeColor(transaction.type)}`}>
                      {transaction.type.toUpperCase()}
                    </span>
                  </div>
                  {transaction.markets && (
                    <p className="text-xs text-black mt-1">
                      Market: {transaction.markets.title}
                    </p>
                  )}
                  <p className="text-xs text-black mt-1">
                    {formatDate(transaction.created_at)}
                  </p>
                </div>
              </div>
              <div className="text-right">
                <div className={`text-lg font-semibold ${getAmountColor(transaction.amount, transaction.type)}`}>
                  {transaction.amount > 0 ? '+' : ''}{formatCurrency(transaction.amount)}
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>

      {transactions.length === 0 && (
        <div className="text-center py-12">
          <div className="text-black mb-4">
            <svg className="mx-auto h-12 w-12" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5H7a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2" />
            </svg>
          </div>
          <h3 className="text-lg font-medium text-black mb-2">No transactions found</h3>
          <p className="text-black">Your transaction history will appear here</p>
        </div>
      )}
    </div>
  )
}
