'use client'

import React, { useState, useEffect } from 'react'
import { accountDeletionService } from '@/lib/services/accountDeletion'
import { useToast } from './Toast'
import { LoadingSpinner } from './LoadingSpinner'

interface DeleteAccountModalProps {
  isOpen: boolean
  onClose: () => void
  userId?: string
  onSuccess?: () => void
}

export default function DeleteAccountModal({ 
  isOpen, 
  onClose, 
  userId, 
  onSuccess 
}: DeleteAccountModalProps) {
  const [step, setStep] = useState<'warning' | 'confirmation' | 'processing'>('warning')
  const [confirmationText, setConfirmationText] = useState('')
  const [reason, setReason] = useState('')
  const [validation, setValidation] = useState<{
    canDelete: boolean
    warnings: string[]
    pendingTransactions: number
    activeBets: number
  } | null>(null)
  const [loading, setLoading] = useState(false)
  const { addToast } = useToast()

  useEffect(() => {
    if (isOpen && userId) {
      validateDeletion()
    }
  }, [isOpen, userId])

  const validateDeletion = async () => {
    if (!userId) return

    try {
      setLoading(true)
      const result = await accountDeletionService.validateAccountDeletion(userId)
      setValidation(result)
    } catch (error) {
      console.error('Error validating deletion:', error)
      addToast({
        type: 'error',
        title: 'Validation Error',
        message: 'Unable to validate account deletion'
      })
    } finally {
      setLoading(false)
    }
  }

  const handleConfirm = () => {
    if (step === 'warning') {
      setStep('confirmation')
    } else if (step === 'confirmation') {
      handleDeleteAccount()
    }
  }

  const handleDeleteAccount = async () => {
    if (!userId) return

    setStep('processing')
    setLoading(true)

    try {
      const result = await accountDeletionService.deleteAccount({
        userId,
        confirmationText,
        reason
      })

      if (result.success) {
        addToast({
          type: 'success',
          title: 'Account Deleted',
          message: 'Your account has been successfully deleted.'
        })
        onSuccess?.()
        onClose()
      } else {
        addToast({
          type: 'error',
          title: 'Deletion Failed',
          message: result.error || 'Failed to delete account'
        })
        setStep('confirmation')
      }
    } catch (error) {
      console.error('Error deleting account:', error)
      addToast({
        type: 'error',
        title: 'Deletion Failed',
        message: 'An unexpected error occurred'
      })
      setStep('confirmation')
    } finally {
      setLoading(false)
    }
  }

  const handleClose = () => {
    if (step === 'processing') return
    setStep('warning')
    setConfirmationText('')
    setReason('')
    setValidation(null)
    onClose()
  }

  if (!isOpen) return null

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-lg shadow-xl max-w-md w-full max-h-[90vh] overflow-y-auto">
        {step === 'warning' && (
          <div className="p-6">
            <div className="flex items-center justify-between mb-4">
              <h2 className="text-xl font-bold text-red-600">Delete Account</h2>
              <button
                onClick={handleClose}
                className="text-gray-400 hover:text-gray-600"
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            </div>

            <div className="mb-6">
              <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-4">
                <div className="flex items-center mb-2">
                  <svg className="w-5 h-5 text-red-600 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z" />
                  </svg>
                  <h3 className="font-semibold text-red-800">Warning</h3>
                </div>
                <p className="text-red-700 text-sm">
                  This action cannot be undone. All your data will be permanently deleted.
                </p>
              </div>

              {loading ? (
                <div className="text-center py-4">
                  <LoadingSpinner size="sm" />
                  <p className="text-gray-600 mt-2">Validating account...</p>
                </div>
              ) : validation && (
                <div className="space-y-3">
                  {validation.warnings.length > 0 && (
                    <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                      <h4 className="font-medium text-yellow-800 mb-2">Before deletion:</h4>
                      <ul className="text-yellow-700 text-sm space-y-1">
                        {validation.warnings.map((warning, index) => (
                          <li key={index} className="flex items-start">
                            <span className="mr-2">•</span>
                            <span>{warning}</span>
                          </li>
                        ))}
                      </ul>
                    </div>
                  )}

                  <div className="text-sm text-gray-600">
                    <p><strong>Pending transactions:</strong> {validation.pendingTransactions}</p>
                    <p><strong>Active bets:</strong> {validation.activeBets}</p>
                  </div>
                </div>
              )}
            </div>

            <div className="flex justify-end space-x-3">
              <button
                onClick={handleClose}
                className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
              >
                Cancel
              </button>
              <button
                onClick={handleConfirm}
                disabled={loading || !validation?.canDelete}
                className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Continue
              </button>
            </div>
          </div>
        )}

        {step === 'confirmation' && (
          <div className="p-6">
            <div className="flex items-center justify-between mb-4">
              <h2 className="text-xl font-bold text-red-600">Confirm Deletion</h2>
              <button
                onClick={handleClose}
                className="text-gray-400 hover:text-gray-600"
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            </div>

            <div className="mb-6">
              <p className="text-gray-700 mb-4">
                To confirm account deletion, please type <strong>"delete my account"</strong> in the box below:
              </p>

              <input
                type="text"
                value={confirmationText}
                onChange={(e) => setConfirmationText(e.target.value)}
                placeholder="Type: delete my account"
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-red-500 focus:border-transparent"
              />

              <div className="mt-4">
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Reason for deletion (optional)
                </label>
                <textarea
                  value={reason}
                  onChange={(e) => setReason(e.target.value)}
                  placeholder="Tell us why you're deleting your account..."
                  rows={3}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-red-500 focus:border-transparent"
                />
              </div>
            </div>

            <div className="flex justify-end space-x-3">
              <button
                onClick={() => setStep('warning')}
                className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
              >
                Back
              </button>
              <button
                onClick={handleConfirm}
                disabled={confirmationText.toLowerCase() !== 'delete my account'}
                className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Delete Account
              </button>
            </div>
          </div>
        )}

        {step === 'processing' && (
          <div className="p-6">
            <div className="text-center">
              <LoadingSpinner size="lg" />
              <h2 className="text-xl font-bold text-red-600 mt-4">Deleting Account</h2>
              <p className="text-gray-600 mt-2">
                Please wait while we process your account deletion...
              </p>
              <div className="mt-4 text-sm text-gray-500">
                <p>• Ending all active sessions</p>
                <p>• Cancelling pending bets</p>
                <p>• Processing withdrawals</p>
                <p>• Removing all data</p>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
