'use client'

import { useState, useRef } from 'react'
import { supabase } from '@/lib/supabase-client'

interface ImageUploadProps {
  value: string
  onChange: (url: string) => void
  onError?: (error: string) => void
  placeholder?: string
  className?: string
  disabled?: boolean
}

export default function ImageUpload({
  value,
  onChange,
  onError,
  placeholder = "Upload an image or enter URL",
  className = "",
  disabled = false
}: ImageUploadProps) {
  const [uploading, setUploading] = useState(false)
  const [preview, setPreview] = useState<string | null>(value || null)
  const fileInputRef = useRef<HTMLInputElement>(null)

  const handleFileUpload = async (file: File) => {
    if (!file) return

    // Validate file type
    if (!file.type.startsWith('image/')) {
      onError?.('Please select an image file')
      return
    }

    // Validate file size (max 5MB)
    if (file.size > 5 * 1024 * 1024) {
      onError?.('Image must be smaller than 5MB')
      return
    }

    setUploading(true)
    onError?.('')

    try {
      // Create a unique filename
      const fileExt = file.name.split('.').pop()
      const fileName = `${Date.now()}-${Math.random().toString(36).substring(2)}.${fileExt}`
      const filePath = `market-images/${fileName}`

      // Upload to Supabase Storage
      const { data, error } = await supabase.storage
        .from('market-images')
        .upload(filePath, file, {
          cacheControl: '3600',
          upsert: false
        })

      if (error) {
        throw error
      }

      // Get the public URL
      const { data: { publicUrl } } = supabase.storage
        .from('market-images')
        .getPublicUrl(filePath)

      // Update the form value
      onChange(publicUrl)
      setPreview(publicUrl)

    } catch (error) {
      console.error('Error uploading image:', error)
      onError?.(error instanceof Error ? error.message : 'Failed to upload image')
    } finally {
      setUploading(false)
    }
  }

  const handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]
    if (file) {
      handleFileUpload(file)
    }
  }

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault()
    const file = e.dataTransfer.files[0]
    if (file) {
      handleFileUpload(file)
    }
  }

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault()
  }

  const handleUrlChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const url = e.target.value
    onChange(url)
    setPreview(url || null)
  }

  const removeImage = () => {
    onChange('')
    setPreview(null)
    if (fileInputRef.current) {
      fileInputRef.current.value = ''
    }
  }

  const openFileDialog = () => {
    fileInputRef.current?.click()
  }

  return (
    <div className={`space-y-3 ${className}`}>
      {/* Upload Area */}
      <div
        className={`border-2 border-dashed rounded-lg p-6 text-center transition-colors cursor-pointer ${
          uploading
            ? 'border-blue-400 bg-blue-50'
            : preview
            ? 'border-green-400 bg-green-50'
            : 'border-gray-300 hover:border-gray-400'
        } ${disabled ? 'opacity-50 cursor-not-allowed' : ''}`}
        onDrop={handleDrop}
        onDragOver={handleDragOver}
        onClick={disabled ? undefined : openFileDialog}
      >
        <input
          ref={fileInputRef}
          type="file"
          accept="image/*"
          onChange={handleFileSelect}
          className="hidden"
          disabled={disabled}
        />

        {uploading ? (
          <div className="flex flex-col items-center">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mb-2"></div>
            <p className="text-sm text-blue-600">Uploading image...</p>
          </div>
        ) : preview ? (
          <div className="space-y-3">
            <img
              src={preview}
              alt="Market preview"
              className="mx-auto max-h-32 max-w-full rounded-lg object-contain"
              onError={() => setPreview(null)}
            />
            <div className="flex justify-center space-x-2">
              <button
                type="button"
                onClick={(e) => {
                  e.stopPropagation()
                  openFileDialog()
                }}
                disabled={disabled}
                className="text-sm text-blue-600 hover:text-blue-700 disabled:opacity-50"
              >
                Change Image
              </button>
              <button
                type="button"
                onClick={(e) => {
                  e.stopPropagation()
                  removeImage()
                }}
                disabled={disabled}
                className="text-sm text-red-600 hover:text-red-700 disabled:opacity-50"
              >
                Remove
              </button>
            </div>
          </div>
        ) : (
          <div className="flex flex-col items-center">
            <svg className="w-12 h-12 text-gray-400 mb-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
            </svg>
            <p className="text-sm text-gray-600 mb-1">
              Click to upload an image or drag and drop
            </p>
            <p className="text-xs text-gray-500">
              PNG, JPG, GIF up to 5MB
            </p>
          </div>
        )}
      </div>

      {/* URL Input */}
      <div className="relative">
        <input
          type="url"
          value={value}
          onChange={handleUrlChange}
          placeholder={placeholder}
          disabled={disabled}
          className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors text-black"
          style={{ color: 'black' }}
        />
        <div className="absolute inset-y-0 right-0 flex items-center pr-3">
          <span className="text-gray-400 text-sm">or paste URL</span>
        </div>
      </div>

      {/* Help Text */}
      <p className="text-xs text-gray-600">
        Upload an image file or enter an image URL. Images make your market more engaging and help attract traders.
      </p>
    </div>
  )
}
