'use client'

import React, { Suspense, lazy, ComponentType } from 'react'
import { LoadingSpinner } from './LoadingSpinner'

interface LazyWrapperProps {
  fallback?: React.ReactNode
}

// Higher-order component for lazy loading
export function withLazyLoading<P extends object>(
  Component: ComponentType<P>,
  fallback?: React.ReactNode
) {
  const LazyComponent = lazy(() => Promise.resolve({ default: Component }))

  return function LazyWrapper(props: P & LazyWrapperProps) {
    return (
      <Suspense fallback={fallback || <LoadingSpinner />}>
        <LazyComponent {...props} />
      </Suspense>
    )
  }
}

// Lazy wrapper for components that might cause performance issues
export function LazyWrapper({ 
  children, 
  fallback 
}: { 
  children: React.ReactNode
  fallback?: React.ReactNode 
}) {
  return (
    <Suspense fallback={fallback || <LoadingSpinner />}>
      {children}
    </Suspense>
  )
}

// Preload components for better performance
export function preloadComponent(importFn: () => Promise<any>) {
  if (typeof window !== 'undefined') {
    // Preload during idle time
    if ('requestIdleCallback' in window) {
      requestIdleCallback(() => {
        importFn()
      })
    } else {
      // Fallback for browsers without requestIdleCallback
      setTimeout(() => {
        importFn()
      }, 100)
    }
  }
}

// Lazy load heavy admin components
export const LazyAdminDashboard = withLazyLoading(
  lazy(() => import('@/components/admin/AdminDashboard')),
  <div className="flex items-center justify-center p-8">
    <LoadingSpinner />
    <span className="ml-2">Loading admin dashboard...</span>
  </div>
)

// TODO: Create AdminAnalytics component
// export const LazyAdminAnalytics = withLazyLoading(
//   lazy(() => import('@/components/admin/AdminAnalytics')),
//   <div className="flex items-center justify-center p-8">
//     <LoadingSpinner />
//     <span className="ml-2">Loading analytics...</span>
//   </div>
// )

export const LazyMarketChart = withLazyLoading(
  lazy(() => import('@/components/markets/MarketChart')),
  <div className="h-64 bg-gray-100 rounded-lg animate-pulse flex items-center justify-center">
    <span className="text-gray-500">Loading chart...</span>
  </div>
)

// Preload commonly used components
if (typeof window !== 'undefined') {
  preloadComponent(() => import('@/components/ui/RecentActivity'))
  preloadComponent(() => import('@/components/ui/TransactionItem'))
  preloadComponent(() => import('@/components/ui/Toast'))
}
