'use client'

import React from 'react'

interface LoadingSpinnerProps {
  size?: 'sm' | 'md' | 'lg' | 'xl'
  color?: 'primary' | 'white' | 'gray'
  text?: string
  fullScreen?: boolean
  overlay?: boolean
}

export function LoadingSpinner({ 
  size = 'md', 
  color = 'primary', 
  text,
  fullScreen = false,
  overlay = false 
}: LoadingSpinnerProps) {
  const sizeClasses = {
    sm: 'w-4 h-4',
    md: 'w-8 h-8',
    lg: 'w-12 h-12',
    xl: 'w-16 h-16'
  }

  const colorClasses = {
    primary: 'text-blue-600',
    white: 'text-white',
    gray: 'text-gray-600'
  }

  const spinner = (
    <div className="flex flex-col items-center justify-center">
      <div className={`animate-spin rounded-full border-2 border-gray-300 border-t-current ${sizeClasses[size]} ${colorClasses[color]}`}></div>
      {text && (
        <p className={`mt-2 text-sm ${colorClasses[color]}`}>
          {text}
        </p>
      )}
    </div>
  )

  if (fullScreen) {
    return (
      <div className="fixed inset-0 bg-white bg-opacity-90 flex items-center justify-center z-50">
        {spinner}
      </div>
    )
  }

  if (overlay) {
    return (
      <div className="absolute inset-0 bg-white bg-opacity-80 flex items-center justify-center z-10">
        {spinner}
      </div>
    )
  }

  return spinner
}

// Specialized loading components for different scenarios
export function PageLoadingSpinner({ text = "Loading..." }: { text?: string }) {
  return (
    <div className="min-h-screen bg-gray-50 flex items-center justify-center">
      <LoadingSpinner size="xl" text={text} />
    </div>
  )
}

export function ButtonLoadingSpinner({ text = "Processing..." }: { text?: string }) {
  return (
    <div className="flex items-center justify-center space-x-2">
      <LoadingSpinner size="sm" color="white" />
      <span className="text-sm">{text}</span>
    </div>
  )
}

export function CardLoadingSpinner({ text = "Loading data..." }: { text?: string }) {
  return (
    <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-8">
      <LoadingSpinner size="lg" text={text} />
    </div>
  )
}

export function TableLoadingSpinner({ rows = 5 }: { rows?: number }) {
  return (
    <div className="space-y-3">
      {Array.from({ length: rows }).map((_, i) => (
        <div key={i} className="flex items-center space-x-4 p-4 bg-white rounded-lg border border-gray-200">
          <div className="w-8 h-8 bg-gray-200 rounded-full animate-pulse"></div>
          <div className="flex-1 space-y-2">
            <div className="h-4 bg-gray-200 rounded animate-pulse"></div>
            <div className="h-3 bg-gray-200 rounded animate-pulse w-2/3"></div>
          </div>
          <div className="w-16 h-4 bg-gray-200 rounded animate-pulse"></div>
        </div>
      ))}
    </div>
  )
}

export function MarketCardLoadingSkeleton() {
  return (
    <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 animate-pulse">
      <div className="flex justify-between items-start mb-4">
        <div className="flex-1">
          <div className="h-4 bg-gray-200 rounded mb-2"></div>
          <div className="h-3 bg-gray-200 rounded w-3/4"></div>
        </div>
        <div className="w-6 h-6 bg-gray-200 rounded"></div>
      </div>
      
      <div className="space-y-3 mb-4">
        <div className="flex justify-between">
          <div className="h-3 bg-gray-200 rounded w-1/4"></div>
          <div className="h-6 bg-gray-200 rounded w-16"></div>
        </div>
        <div className="flex justify-between">
          <div className="h-3 bg-gray-200 rounded w-1/4"></div>
          <div className="h-6 bg-gray-200 rounded w-16"></div>
        </div>
      </div>
      
      <div className="flex justify-between items-center">
        <div className="h-3 bg-gray-200 rounded w-1/3"></div>
        <div className="h-8 bg-gray-200 rounded w-20"></div>
      </div>
    </div>
  )
}

export function BetFormLoadingSpinner({ text = "Processing bet..." }: { text?: string }) {
  return (
    <div className="bg-white rounded-lg border border-gray-200 p-6">
      <LoadingSpinner size="lg" text={text} />
    </div>
  )
}

export function TransactionLoadingSpinner({ text = "Processing transaction..." }: { text?: string }) {
  return (
    <div className="bg-white rounded-lg border border-gray-200 p-4">
      <LoadingSpinner size="md" text={text} />
    </div>
  )
}