'use client'

import React, { memo, useState } from 'react'
import { useLoginSessions } from '@/lib/hooks/useLoginSessions'
import { formatSessionInfo, formatTimeAgo } from '@/lib/utils/deviceDetection'
import { LoadingSpinner } from './LoadingSpinner'

interface LoginActivityProps {
  userId?: string
  className?: string
}

const SessionItem = memo(({ session, isCurrentSession = false }: { 
  session: any, 
  isCurrentSession?: boolean 
}) => {
  const [showDetails, setShowDetails] = useState(false)
  const getStatusColor = (isActive: boolean) => {
    return isActive ? 'text-green-600' : 'text-black'
  }

  const getStatusText = (isActive: boolean) => {
    return isActive ? 'Active' : 'Ended'
  }

  const getDeviceIcon = (deviceType?: string) => {
    switch (deviceType) {
      case 'mobile':
        return '📱'
      case 'tablet':
        return '📱'
      default:
        return '💻'
    }
  }

  return (
    <div className="flex items-center justify-between py-3 border-b border-gray-100 last:border-b-0">
      <div className="flex items-center space-x-3">
        <div className="text-2xl">
          {getDeviceIcon(session.device_type)}
        </div>
        <div>
          <div className="font-medium text-black">
            {isCurrentSession ? 'Current Session' : formatSessionInfo(session)}
          </div>
          <div className="text-sm text-black">
            {formatTimeAgo(session.login_time)}
            {session.last_activity && session.last_activity !== session.login_time && (
              <span className="text-gray-600">
                {' • Last active '}{formatTimeAgo(session.last_activity)}
              </span>
            )}
          </div>
          {session.location && (
            <div className="text-sm text-blue-600 font-medium">
              📍 {session.location}
            </div>
          )}
          <div className="text-xs text-black font-mono">
            🌐 {session.ip_address || 'IP not captured'}
          </div>
          <button
            onClick={() => setShowDetails(!showDetails)}
            className="text-xs text-blue-600 hover:text-blue-800 mt-1"
          >
            {showDetails ? 'Hide Details' : 'Show Details'}
          </button>
          {showDetails && (
            <div className="mt-2 p-3 bg-gray-50 rounded-lg text-xs space-y-1 text-black">
              <div><strong>Browser:</strong> {session.browser || 'Unknown'}</div>
              <div><strong>OS:</strong> {session.os || 'Unknown'}</div>
              <div><strong>Device:</strong> {session.device_type || 'Unknown'}</div>
              <div><strong>User Agent:</strong> {session.user_agent || 'Unknown'}</div>
              <div><strong>Login Time:</strong> {new Date(session.login_time).toLocaleString()}</div>
              {session.last_activity && (
                <div><strong>Last Activity:</strong> {new Date(session.last_activity).toLocaleString()}</div>
              )}
              {session.logout_time && (
                <div><strong>Logout Time:</strong> {new Date(session.logout_time).toLocaleString()}</div>
              )}
            </div>
          )}
        </div>
      </div>
      <div className={`text-sm font-medium ${getStatusColor(session.is_active)}`}>
        {getStatusText(session.is_active)}
      </div>
    </div>
  )
})

SessionItem.displayName = 'SessionItem'

const LoadingState = memo(() => (
  <div className="space-y-3">
    {Array.from({ length: 3 }).map((_, index) => (
      <div key={index} className="animate-pulse">
        <div className="flex items-center space-x-3 py-3">
          <div className="w-8 h-8 bg-gray-300 rounded"></div>
          <div className="flex-1">
            <div className="h-4 bg-gray-300 rounded w-3/4 mb-2"></div>
            <div className="h-3 bg-gray-300 rounded w-1/2"></div>
          </div>
          <div className="w-16 h-3 bg-gray-300 rounded"></div>
        </div>
      </div>
    ))}
  </div>
))

LoadingState.displayName = 'LoadingState'

const EmptyState = memo(() => (
  <div className="text-center py-8">
    <div className="text-black text-4xl mb-2">🔐</div>
    <p className="text-black">No login activity found</p>
    <p className="text-black text-sm">Your login sessions will appear here</p>
  </div>
))

EmptyState.displayName = 'EmptyState'

const ErrorState = memo(({ error }: { error: string }) => (
  <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
    <p className="text-red-600 text-sm">Error loading login activity: {error}</p>
  </div>
))

ErrorState.displayName = 'ErrorState'

function LoginActivity({ userId, className = '' }: LoginActivityProps) {
  console.log('LoginActivity: Component rendered', { userId })
  const { 
    sessions, 
    loading, 
    error, 
    getCurrentSession, 
    getActiveSessions, 
    getInactiveSessions 
  } = useLoginSessions(userId)
  
  console.log('LoginActivity: Hook results', { sessions, loading, error })

  if (loading) {
    return <LoadingState />
  }

  if (error) {
    return <ErrorState error={error} />
  }

  if (sessions.length === 0) {
    return <EmptyState />
  }

  const currentSession = getCurrentSession()
  const activeSessions = getActiveSessions()
  const inactiveSessions = getInactiveSessions()
  
  // Calculate location statistics
  const locationStats = sessions.reduce((acc: any, session: any) => {
    if (session.location && session.location !== 'Unknown Location') {
      acc[session.location] = (acc[session.location] || 0) + 1
    }
    return acc
  }, {})
  
  const uniqueLocations = Object.keys(locationStats).length
  const mostCommonLocation = Object.entries(locationStats).sort(([,a], [,b]) => (b as number) - (a as number))[0]

  return (
    <div className={`space-y-3 ${className}`}>
      {/* Location Statistics */}
      {uniqueLocations > 0 && (
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-4">
          <h4 className="text-sm font-medium text-blue-900 mb-2">📍 Location Summary</h4>
          <div className="text-sm text-blue-800">
            <div>• {uniqueLocations} unique location{uniqueLocations !== 1 ? 's' : ''} detected</div>
            {mostCommonLocation && (
              <div>• Most common: {String(mostCommonLocation[0])} ({String(mostCommonLocation[1])} session{Number(mostCommonLocation[1]) !== 1 ? 's' : ''})</div>
            )}
          </div>
        </div>
      )}
      
      {/* Current Session */}
      {currentSession && (
        <SessionItem 
          session={currentSession} 
          isCurrentSession={true} 
        />
      )}

      {/* Other Active Sessions */}
      {activeSessions
        .filter(session => session.id !== currentSession?.id)
        .map((session) => (
          <SessionItem 
            key={session.id} 
            session={session} 
          />
        ))}

      {/* Recent Inactive Sessions */}
      {inactiveSessions.slice(0, 5).map((session) => (
        <SessionItem 
          key={session.id} 
          session={session} 
        />
      ))}
    </div>
  )
}

export default memo(LoginActivity)
