'use client'

import React, { memo } from 'react'
import { useRecentActivityOptimized } from '@/lib/hooks/useRecentActivityOptimized'
import { useAuth } from '@/components/auth/ClientAuthProvider'
import Link from 'next/link'

interface RecentActivityProps {
  limit?: number
  showMarketLinks?: boolean
  className?: string
}

const ActivityItem = memo(({ activity, showMarketLinks }: { 
  activity: any, 
  showMarketLinks: boolean 
}) => {
  const getActivityIcon = (type: string) => {
    switch (type) {
      case 'bet':
        return '🎯'
      case 'deposit':
        return '💰'
      case 'withdrawal':
        return '💸'
      case 'payout':
        return '🎉'
      case 'seed':
        return '🌱'
      case 'liquidation':
        return '💧'
      default:
        return '📊'
    }
  }

  const getActivityColor = (type: string) => {
    switch (type) {
      case 'bet':
        return 'text-blue-600 bg-blue-100'
      case 'deposit':
        return 'text-green-600 bg-green-100'
      case 'withdrawal':
        return 'text-red-600 bg-red-100'
      case 'payout':
        return 'text-purple-600 bg-purple-100'
      case 'seed':
        return 'text-yellow-600 bg-yellow-100'
      case 'liquidation':
        return 'text-orange-600 bg-orange-100'
      default:
        return 'text-gray-600 bg-gray-100'
    }
  }

  return (
    <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors">
      <div className="flex items-center space-x-3">
        <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm ${getActivityColor(activity.type)}`}>
          {getActivityIcon(activity.type)}
        </div>
        <div>
          <div className="font-medium text-black">
            {activity.description}
          </div>
          <div className="text-sm text-gray-600">
            {new Date(activity.timestamp).toLocaleString()}
          </div>
        </div>
      </div>
      
      <div className="flex items-center space-x-2">
        {activity.amount && (
          <span className="text-sm font-medium text-black">
            ${activity.amount.toFixed(2)}
          </span>
        )}
        {activity.status && (
          <span className={`px-2 py-1 text-xs rounded-full ${
            activity.status === 'completed' || activity.status === 'won' ? 'bg-green-100 text-green-800' :
            activity.status === 'pending' ? 'bg-yellow-100 text-yellow-800' :
            activity.status === 'failed' || activity.status === 'lost' ? 'bg-red-100 text-red-800' :
            'bg-gray-100 text-gray-800'
          }`}>
            {activity.status}
          </span>
        )}
        {showMarketLinks && activity.market_id && activity.market_title && (
          <Link 
            href={`/markets/${activity.market_id}`}
            className="text-blue-600 hover:text-blue-800 text-sm underline"
          >
            View Market
          </Link>
        )}
      </div>
    </div>
  )
})

ActivityItem.displayName = 'ActivityItem'

const LoadingSkeleton = memo(({ limit }: { limit: number }) => (
  <div className="space-y-3">
    {Array.from({ length: Math.min(limit, 5) }).map((_, index) => (
      <div key={index} className="animate-pulse">
        <div className="flex items-center space-x-3 p-3 bg-gray-50 rounded-lg">
          <div className="w-8 h-8 bg-gray-300 rounded-full"></div>
          <div className="flex-1">
            <div className="h-4 bg-gray-300 rounded w-3/4 mb-2"></div>
            <div className="h-3 bg-gray-300 rounded w-1/2"></div>
          </div>
          <div className="w-16 h-3 bg-gray-300 rounded"></div>
        </div>
      </div>
    ))}
  </div>
))

LoadingSkeleton.displayName = 'LoadingSkeleton'

const EmptyState = memo(() => (
  <div className="p-6 text-center">
    <div className="text-gray-400 text-4xl mb-2">📊</div>
    <p className="text-gray-500">No recent activity</p>
    <p className="text-gray-400 text-sm">Your recent transactions and bets will appear here</p>
  </div>
))

EmptyState.displayName = 'EmptyState'

const ErrorState = memo(({ error }: { error: string }) => (
  <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
    <p className="text-red-600 text-sm">Error loading recent activity: {error}</p>
  </div>
))

ErrorState.displayName = 'ErrorState'

function RecentActivityOptimized({ 
  limit = 10, 
  showMarketLinks = true,
  className = ''
}: RecentActivityProps) {
  const { user } = useAuth()
  const { activities, loading, error } = useRecentActivityOptimized(user?.id, limit)

  if (loading) {
    return <LoadingSkeleton limit={limit} />
  }

  if (error) {
    return <ErrorState error={error} />
  }

  if (activities.length === 0) {
    return <EmptyState />
  }

  return (
    <div className={`space-y-3 ${className}`}>
      {activities.map((activity) => (
        <ActivityItem 
          key={activity.id} 
          activity={activity} 
          showMarketLinks={showMarketLinks} 
        />
      ))}
    </div>
  )
}

export default memo(RecentActivityOptimized)
