'use client'

import Link from 'next/link'

interface TransactionItemProps {
  transaction: {
    id: string
    type: string
    status: string
    amount: number
    description?: string | null
    created_at: string
    market_id?: string | null
  }
  showMarketLink?: boolean
}

export default function TransactionItem({ transaction, showMarketLink = true }: TransactionItemProps) {
  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  // Determine transaction type styling and icon
  const getTransactionInfo = (type: string) => {
    switch (type) {
      case 'deposit':
        return { color: 'text-green-600', bg: 'bg-green-50', icon: '💰', label: 'Deposit' }
      case 'withdrawal':
        return { color: 'text-red-600', bg: 'bg-red-50', icon: '💸', label: 'Withdrawal' }
      case 'bet':
        return { color: 'text-blue-600', bg: 'bg-blue-50', icon: '🎯', label: 'Bet Placed' }
      case 'win':
        return { color: 'text-green-600', bg: 'bg-green-50', icon: '🏆', label: 'Bet Won' }
      case 'seeding':
        return { color: 'text-purple-600', bg: 'bg-purple-50', icon: '🌱', label: 'Market Seeded' }
      case 'liquidity':
        return { color: 'text-indigo-600', bg: 'bg-indigo-50', icon: '💧', label: 'Liquidity Added' }
      case 'market_creation':
        return { color: 'text-orange-600', bg: 'bg-orange-50', icon: '🏗️', label: 'Market Created' }
      case 'refund':
        return { color: 'text-yellow-600', bg: 'bg-yellow-50', icon: '↩️', label: 'Refund' }
      case 'fee':
        return { color: 'text-gray-600', bg: 'bg-gray-50', icon: '💳', label: 'Fee' }
      case 'bonus':
        return { color: 'text-pink-600', bg: 'bg-pink-50', icon: '🎁', label: 'Bonus' }
      case 'transfer_in':
        return { color: 'text-green-600', bg: 'bg-green-50', icon: '⬇️', label: 'Transfer In' }
      case 'transfer_out':
        return { color: 'text-red-600', bg: 'bg-red-50', icon: '⬆️', label: 'Transfer Out' }
      default:
        return { color: 'text-gray-600', bg: 'bg-gray-50', icon: '📄', label: type }
    }
  }

  const transactionInfo = getTransactionInfo(transaction.type)
  
  return (
    <div className="flex items-center justify-between py-3 border-b border-gray-100 last:border-b-0">
      <div className="flex items-center space-x-3 flex-1">
        <div className={`w-8 h-8 rounded-full ${transactionInfo.bg} flex items-center justify-center text-sm`}>
          {transactionInfo.icon}
        </div>
        <div className="flex-1">
          <div className="flex items-center space-x-2">
            <div className="font-medium text-black">
              {transactionInfo.label}
            </div>
            <div className={`px-2 py-1 rounded-full text-xs font-medium ${transactionInfo.bg} ${transactionInfo.color}`}>
              {transaction.status}
            </div>
          </div>
          <div className="text-sm text-gray-600">
            {transaction.description || transaction.type}
          </div>
          <div className="text-xs text-gray-500">
            {new Date(transaction.created_at).toLocaleDateString()} at {new Date(transaction.created_at).toLocaleTimeString()}
          </div>
          {showMarketLink && transaction.market_id && (
            <Link 
              href={`/markets/${transaction.market_id}`}
              className="text-xs text-blue-600 hover:text-blue-700"
            >
              View Market →
            </Link>
          )}
        </div>
      </div>
      <div className={`font-semibold ${transaction.amount > 0 ? 'text-green-600' : 'text-red-600'}`}>
        {transaction.amount > 0 ? '+' : ''}{formatCurrency(transaction.amount)}
      </div>
    </div>
  )
}
