'use client'

import { Profile, DailyLimits } from '@/lib/types'

interface DailyLimitsCardProps {
  profile: Profile
  dailyLimits: DailyLimits | null
}

export default function DailyLimitsCard({ profile, dailyLimits }: DailyLimitsCardProps) {
  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  const getUsagePercentage = (used: number, limit: number) => {
    return Math.min((used / limit) * 100, 100)
  }

  const getProgressColor = (percentage: number) => {
    if (percentage >= 90) return 'bg-red-500'
    if (percentage >= 70) return 'bg-yellow-500'
    return 'bg-green-500'
  }

  const depositsUsed = dailyLimits?.deposits_amount || 0
  const withdrawalsUsed = dailyLimits?.withdrawals_amount || 0

  const depositsPercentage = getUsagePercentage(depositsUsed, profile.daily_deposit_limit)
  const withdrawalsPercentage = getUsagePercentage(withdrawalsUsed, profile.daily_withdrawal_limit)

  return (
    <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
      <h3 className="text-lg font-semibold text-black mb-4">Daily Limits</h3>
      
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        {/* Deposits */}
        <div>
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-black">Deposits</span>
            <span className="text-sm text-black">
              {formatCurrency(depositsUsed)} / {formatCurrency(profile.daily_deposit_limit)}
            </span>
          </div>
          <div className="w-full bg-gray-200 rounded-full h-2 mb-1">
            <div
              className={`h-2 rounded-full ${getProgressColor(depositsPercentage)}`}
              style={{ width: `${depositsPercentage}%` }}
            />
          </div>
          <div className="flex justify-between text-xs text-black">
            <span>{dailyLimits?.deposits_count || 0} transactions</span>
            <span>{depositsPercentage.toFixed(1)}% used</span>
          </div>
        </div>

        {/* Withdrawals */}
        <div>
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium text-black">Withdrawals</span>
            <span className="text-sm text-black">
              {formatCurrency(withdrawalsUsed)} / {formatCurrency(profile.daily_withdrawal_limit)}
            </span>
          </div>
          <div className="w-full bg-gray-200 rounded-full h-2 mb-1">
            <div
              className={`h-2 rounded-full ${getProgressColor(withdrawalsPercentage)}`}
              style={{ width: `${withdrawalsPercentage}%` }}
            />
          </div>
          <div className="flex justify-between text-xs text-black">
            <span>{dailyLimits?.withdrawals_count || 0} transactions</span>
            <span>{withdrawalsPercentage.toFixed(1)}% used</span>
          </div>
        </div>
      </div>

      {/* Monthly Limits Info */}
      <div className="mt-6 pt-4 border-t border-gray-200">
        <div className="grid grid-cols-2 gap-4 text-center">
          <div>
            <div className="text-lg font-semibold text-black">
              {formatCurrency(profile.monthly_deposit_limit)}
            </div>
            <div className="text-sm text-black">Monthly Deposit Limit</div>
          </div>
          <div>
            <div className="text-lg font-semibold text-black">
              {formatCurrency(profile.monthly_withdrawal_limit)}
            </div>
            <div className="text-sm text-black">Monthly Withdrawal Limit</div>
          </div>
        </div>
      </div>

      {/* Limits Reset Info */}
      <div className="mt-4 p-3 bg-blue-50 rounded-lg">
        <div className="flex items-center space-x-2">
          <svg className="w-5 h-5 text-blue-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
          </svg>
          <span className="text-sm text-blue-700">
            Daily limits reset at midnight UTC
          </span>
        </div>
      </div>
    </div>
  )
}
