'use client'

import { useState } from 'react'
import { Profile, PaymentMethod } from '@/lib/types'
import { useToast } from '@/components/ui/Toast'
import { LoadingSpinner, ButtonLoadingSpinner, TransactionLoadingSpinner } from '@/components/ui/LoadingSpinner'
import { unifiedPaymentService, PaymentProvider } from '@/lib/payments'

interface DepositModalProps {
  profile: Profile
  paymentMethods: PaymentMethod[]
  onClose: () => void
}

export default function DepositModal({ profile, paymentMethods, onClose }: DepositModalProps) {
  const [amount, setAmount] = useState('')
  const [selectedMethod, setSelectedMethod] = useState<string>('')
  const [selectedProvider, setSelectedProvider] = useState<PaymentProvider>('paynow_zim')
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState('')
  const [success, setSuccess] = useState(false)
  const { addToast } = useToast()
  
  // USD-only implementation
  const currency = 'USD'

  const formatCurrency = (value: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: currency
    }).format(value)
  }

  // Get supported payment methods for current currency
  const getSupportedPaymentMethods = () => {
    return unifiedPaymentService.getSupportedPaymentMethods(currency)
  }

  const handleDeposit = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)
    setError('')

    try {
      const depositAmount = parseFloat(amount)
      
      if (!depositAmount || depositAmount <= 0) {
        setError('Please enter a valid amount')
        return
      }

      if (depositAmount < 10) {
        setError('Minimum deposit amount is $10')
        setLoading(false)
        return
      }

      if (depositAmount > profile.daily_deposit_limit) {
        setError(`Deposit amount exceeds daily limit of ${formatCurrency(profile.daily_deposit_limit)}`)
        setLoading(false)
        return
      }

      if (depositAmount > 10000) {
        setError('Maximum single deposit amount is $10,000')
        setLoading(false)
        return
      }

      // Create payment request
      const paymentRequest = {
        provider: selectedProvider,
        amount: depositAmount,
        currency: currency,
        description: `Deposit to BetFusion Zim account`,
        reference: `deposit_${profile.id}_${Date.now()}`,
        customerEmail: profile.email,
        customerPhone: profile.phone || undefined,
        customerFirstName: profile.first_name || undefined,
        customerLastName: profile.last_name || undefined,
        returnUrl: `${window.location.origin}/wallet?deposit=success`,
        cancelUrl: `${window.location.origin}/wallet?deposit=cancelled`
      }

      // Create transaction record first
      const { supabase } = await import('@/lib/supabase-client')
      const { data: transaction, error: transactionError } = await supabase
        .from('transactions')
        .insert({
          user_id: profile.id,
          type: 'deposit',
          amount: depositAmount,
          currency: currency,
          status: 'pending',
          payment_provider: selectedProvider,
          reference: paymentRequest.reference,
          description: paymentRequest.description
        })
        .select()
        .single()

      if (transactionError) {
        throw new Error('Failed to create transaction record')
      }

      // Process payment with selected provider
      const paymentResponse = await unifiedPaymentService.createPayment(paymentRequest)

      if (!paymentResponse.success) {
        throw new Error(paymentResponse.error || 'Payment creation failed')
      }

      // Update transaction with external ID
      if (paymentResponse.transactionId) {
        await supabase
          .from('transactions')
          .update({ external_transaction_id: paymentResponse.transactionId })
          .eq('id', transaction.id)
      }

      // Redirect to payment provider
      if (paymentResponse.paymentUrl) {
        window.location.href = paymentResponse.paymentUrl
      } else {
        throw new Error('No payment URL received from provider')
      }

    } catch (err) {
      console.error('Deposit error:', err)
      const errorMessage = err instanceof Error ? err.message : 'An error occurred. Please try again.'
      setError(errorMessage)
      addToast({
        type: 'error',
        title: 'Deposit Failed',
        message: errorMessage
      })
      setLoading(false)
    }
  }

  const quickAmounts = [50, 100, 250, 500, 1000]

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-xl shadow-xl max-w-md w-full max-h-[90vh] overflow-y-auto relative">
        {loading && <LoadingSpinner overlay text="Processing deposit..." />}
        <div className="p-4 sm:p-6">
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-xl sm:text-2xl font-bold text-black">Deposit Funds</h2>
            <button
              onClick={onClose}
              className="text-black hover:text-black transition-colors p-1"
            >
              <svg className="w-5 h-5 sm:w-6 sm:h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
              </svg>
            </button>
          </div>

          <form onSubmit={handleDeposit} className="space-y-4">
            {/* Amount Input */}
            <div>
              <label className="block text-sm font-medium text-black mb-2">
                Deposit Amount
              </label>
              <div className="relative">
                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                  <span className="text-black sm:text-sm">$</span>
                </div>
                <input
                  type="number"
                  step="0.01"
                  min="10"
                  max={Math.min(profile.daily_deposit_limit, 10000)}
                  value={amount}
                  onChange={(e) => {
                    setAmount(e.target.value)
                    setError('') // Clear error when user types
                  }}
                  className="block w-full pl-7 pr-3 py-3 border border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500 !text-black"
                  placeholder="0.00"
                  required
                  disabled={loading}
                />
              </div>
              <div className="mt-1 text-xs sm:text-sm text-black">
                Daily limit: {formatCurrency(profile.daily_deposit_limit)}
              </div>
            </div>

            {/* Quick Amount Buttons */}
            <div>
              <label className="block text-xs sm:text-sm font-medium text-black mb-1 sm:mb-2">
                Quick Amounts
              </label>
              <div className="grid grid-cols-5 gap-1 sm:gap-2">
                {quickAmounts.map((quickAmount) => (
                  <button
                    key={quickAmount}
                    type="button"
                    onClick={() => setAmount(quickAmount.toString())}
                    className="px-2 sm:px-3 py-1.5 sm:py-2 text-xs sm:text-sm font-medium text-black bg-gray-100 hover:bg-gray-200 rounded-lg transition-colors"
                  >
                    ${quickAmount}
                  </button>
                ))}
              </div>
            </div>

            {/* USD-only notice */}
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-2 sm:p-3">
              <div className="flex items-center">
                <div className="text-blue-600 mr-2 text-sm sm:text-base">💵</div>
                <div className="text-xs sm:text-sm text-blue-800">
                  <strong>USD Only:</strong> All transactions are processed in US Dollars
                </div>
              </div>
            </div>

            {/* Payment Provider Selection */}
            <div>
              <label className="block text-xs sm:text-sm font-medium text-black mb-1 sm:mb-2">
                Payment Method
              </label>
              <div className="space-y-2 sm:space-y-3">
                {getSupportedPaymentMethods().map((provider) => {
                  const methodInfo = unifiedPaymentService.getPaymentMethodInfo(provider)
                  return (
                    <div
                      key={provider}
                      className={`border-2 rounded-lg p-3 sm:p-4 cursor-pointer transition-colors ${
                        selectedProvider === provider
                          ? 'border-blue-500 bg-blue-50'
                          : 'border-gray-200 hover:border-gray-300'
                      }`}
                      onClick={() => setSelectedProvider(provider)}
                    >
                      <div className="flex items-center space-x-2 sm:space-x-3">
                        <div className="text-xl sm:text-2xl">{methodInfo.icon}</div>
                        <div className="flex-1">
                          <div className="flex items-center space-x-2">
                            <input
                              type="radio"
                              name="paymentProvider"
                              value={provider}
                              checked={selectedProvider === provider}
                              onChange={() => setSelectedProvider(provider)}
                              className="text-blue-600 focus:ring-blue-500"
                            />
                            <h3 className="text-sm sm:text-base font-medium text-black">{methodInfo.name}</h3>
                          </div>
                          <p className="text-xs sm:text-sm text-gray-600 mt-1">{methodInfo.description}</p>
                          <p className="text-xs text-gray-500 mt-1">Processing time: {methodInfo.processingTime}</p>
                        </div>
                      </div>
                    </div>
                  )
                })}
              </div>
            </div>

            {/* Fee Information */}
            <div className="bg-blue-50 rounded-lg p-2 sm:p-3">
              <div className="flex items-center space-x-2">
                <svg className="w-4 h-4 sm:w-5 sm:h-5 text-blue-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <span className="text-xs sm:text-sm font-medium text-blue-800">Deposit Information</span>
              </div>
              <ul className="mt-1 sm:mt-2 text-xs sm:text-sm text-blue-700 space-y-0.5 sm:space-y-1">
                <li>• No deposit fees</li>
                <li>• Instant processing</li>
                <li>• Funds available immediately</li>
              </ul>
            </div>

            {/* Error Message */}
            {error && (
              <div className="bg-red-50 border border-red-200 rounded-lg p-2 sm:p-3">
                <div className="flex items-start">
                  <svg className="w-4 h-4 sm:w-5 sm:h-5 text-red-400 mt-0.5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                  </svg>
                  <div className="ml-2 sm:ml-3">
                    <h3 className="text-xs sm:text-sm font-medium text-red-800">Deposit Error</h3>
                    <p className="text-xs sm:text-sm text-red-700 mt-1">{error}</p>
                  </div>
                </div>
              </div>
            )}

            {/* Action Buttons */}
            <div className="flex space-x-2 sm:space-x-3">
              <button
                type="button"
                onClick={onClose}
                className="flex-1 bg-gray-100 hover:bg-gray-200 text-black font-medium py-2 sm:py-3 px-3 sm:px-4 rounded-lg transition-colors text-sm sm:text-base"
              >
                Cancel
              </button>
              <button
                type="submit"
                disabled={loading || !amount}
                className="flex-1 bg-blue-600 hover:bg-blue-700 disabled:bg-gray-300 text-white font-medium py-2 sm:py-3 px-3 sm:px-4 rounded-lg transition-colors flex items-center justify-center text-sm sm:text-base"
              >
                {loading ? (
                  <ButtonLoadingSpinner text="Processing deposit..." />
                ) : (
                  `Pay with ${unifiedPaymentService.getPaymentMethodInfo(selectedProvider).name}`
                )}
              </button>
            </div>
          </form>
        </div>
      </div>
    </div>
  )
}
