'use client'

import { useState } from 'react'
import { Profile, PaymentMethod, WalletTransaction, DailyLimits } from '@/lib/types'
import BalanceCard from './BalanceCard'
import PaymentMethodsCard from './PaymentMethodsCard'
import TransactionHistory from './TransactionHistory'
import DailyLimitsCard from './DailyLimitsCard'
import DepositModal from './DepositModal'
import WithdrawModal from './WithdrawModal'
import RecentActivity from '@/components/ui/RecentActivity'

interface WalletDashboardProps {
  profile: Profile
  paymentMethods: PaymentMethod[]
  transactions: WalletTransaction[]
  dailyLimits: DailyLimits | null
}

export default function WalletDashboard({ 
  profile, 
  paymentMethods, 
  transactions, 
  dailyLimits 
}: WalletDashboardProps) {
  const [activeTab, setActiveTab] = useState<'overview' | 'transactions' | 'settings'>('overview')
  const [showDepositModal, setShowDepositModal] = useState(false)
  const [showWithdrawModal, setShowWithdrawModal] = useState(false)

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(amount)
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    })
  }

  return (
    <div className="space-y-6">
      {/* Balance Overview */}
      <BalanceCard 
        profile={profile}
        onDeposit={() => setShowDepositModal(true)}
        onWithdraw={() => setShowWithdrawModal(true)}
      />

      {/* Daily Limits */}
      <DailyLimitsCard 
        profile={profile}
        dailyLimits={dailyLimits}
      />

      {/* Tabs */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200">
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6" aria-label="Tabs">
            <button
              onClick={() => setActiveTab('overview')}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === 'overview'
                  ? 'border-blue-500 text-blue-600'
                  : 'border-transparent text-black hover:text-black hover:border-gray-300'
              }`}
            >
              Overview
            </button>
            <button
              onClick={() => setActiveTab('transactions')}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === 'transactions'
                  ? 'border-blue-500 text-blue-600'
                  : 'border-transparent text-black hover:text-black hover:border-gray-300'
              }`}
            >
              Transaction History
            </button>
            <button
              onClick={() => setActiveTab('settings')}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === 'settings'
                  ? 'border-blue-500 text-blue-600'
                  : 'border-transparent text-black hover:text-black hover:border-gray-300'
              }`}
            >
              Payment Methods
            </button>
          </nav>
        </div>

        <div className="p-6">
          {activeTab === 'overview' && (
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Quick Stats */}
              <div className="bg-gradient-to-br from-blue-50 to-indigo-100 rounded-lg p-6">
                <h3 className="text-lg font-semibold text-black mb-4">Quick Stats</h3>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <div className="text-2xl font-bold text-green-600">
                      {formatCurrency(profile.balance)}
                    </div>
                    <div className="text-sm text-black">Available Balance</div>
                  </div>
                  <div>
                    <div className="text-2xl font-bold text-blue-600">
                      {profile.total_points.toLocaleString()}
                    </div>
                    <div className="text-sm text-black">Reward Points</div>
                  </div>
                  <div>
                    <div className="text-2xl font-bold text-purple-600">
                      {paymentMethods.length}
                    </div>
                    <div className="text-sm text-black">Payment Methods</div>
                  </div>
                  <div>
                    <div className="text-2xl font-bold text-orange-600">
                      {profile.kyc_verified ? 'Verified' : 'Pending'}
                    </div>
                    <div className="text-sm text-black">KYC Status</div>
                  </div>
                </div>
              </div>

              {/* Recent Activity */}
              <div>
                <h3 className="text-lg font-semibold text-black mb-4">Recent Activity</h3>
                <RecentActivity limit={5} showMarketLinks={false} />
              </div>
            </div>
          )}

          {activeTab === 'transactions' && (
            <TransactionHistory transactions={transactions} />
          )}

          {activeTab === 'settings' && (
            <PaymentMethodsCard 
              paymentMethods={paymentMethods}
              onAddMethod={() => {/* TODO: Implement add payment method */}}
            />
          )}
        </div>
      </div>

      {/* Modals */}
      {showDepositModal && (
        <DepositModal
          profile={profile}
          paymentMethods={paymentMethods}
          onClose={() => setShowDepositModal(false)}
        />
      )}

      {showWithdrawModal && (
        <WithdrawModal
          profile={profile}
          paymentMethods={paymentMethods}
          onClose={() => setShowWithdrawModal(false)}
        />
      )}
    </div>
  )
}
