'use client'

import React, { useState, memo } from 'react'
import { Profile, PaymentMethod, WalletTransaction, DailyLimits } from '@/lib/types'
import BalanceCard from './BalanceCard'
import PaymentMethodsCard from './PaymentMethodsCard'
import TransactionHistory from './TransactionHistory'
import DailyLimitsCard from './DailyLimitsCard'
import DepositModal from './DepositModal'
import WithdrawModal from './WithdrawModal'
import RecentActivityOptimized from '@/components/ui/RecentActivityOptimized'
import { usePerformanceMonitor } from '@/lib/hooks/usePerformanceMonitor'

interface WalletDashboardProps {
  profile: Profile
  paymentMethods: PaymentMethod[]
  transactions: WalletTransaction[]
  dailyLimits: DailyLimits | null
}

const TabButton = memo(({ 
  tab, 
  isActive, 
  onClick 
}: { 
  tab: { id: string; name: string; icon: string }, 
  isActive: boolean, 
  onClick: () => void 
}) => (
  <button
    onClick={onClick}
    className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors ${
      isActive
        ? 'border-blue-500 text-blue-600'
        : 'border-transparent text-black hover:text-black hover:border-gray-300'
    }`}
  >
    <span className="mr-2">{tab.icon}</span>
    {tab.name}
  </button>
))

TabButton.displayName = 'TabButton'

function WalletDashboardOptimized({ 
  profile, 
  paymentMethods, 
  transactions, 
  dailyLimits 
}: WalletDashboardProps) {
  const [activeTab, setActiveTab] = useState<'overview' | 'transactions' | 'settings'>('overview')
  const [showDepositModal, setShowDepositModal] = useState(false)
  const [showWithdrawModal, setShowWithdrawModal] = useState(false)

  // Monitor performance in development
  usePerformanceMonitor('WalletDashboard', process.env.NODE_ENV === 'development')

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD',
      minimumFractionDigits: 2,
      maximumFractionDigits: 2
    }).format(amount)
  }

  const tabs = [
    { id: 'overview', name: 'Overview', icon: '💰' },
    { id: 'transactions', name: 'Transactions', icon: '📋' },
    { id: 'settings', name: 'Settings', icon: '⚙️' },
  ]

  return (
    <div className="space-y-6">
      {/* Balance and Action Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <BalanceCard 
          profile={profile} 
          onDeposit={() => setShowDepositModal(true)}
          onWithdraw={() => setShowWithdrawModal(true)}
        />
        <DailyLimitsCard profile={profile} dailyLimits={dailyLimits} />
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <div className="text-2xl font-bold text-orange-600">
                {profile.kyc_verified ? 'Verified' : 'Pending'}
              </div>
              <div className="text-sm text-black">KYC Status</div>
            </div>
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <div className="text-2xl font-bold text-blue-600">
                {paymentMethods.length}
              </div>
              <div className="text-sm text-black">Payment Methods</div>
            </div>
          </div>
        </div>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="border-b border-gray-200">
          <nav className="-mb-px flex space-x-8 px-6">
            {tabs.map((tab) => (
              <TabButton
                key={tab.id}
                tab={tab}
                isActive={activeTab === tab.id}
                onClick={() => setActiveTab(tab.id as any)}
              />
            ))}
          </nav>
        </div>

        <div className="p-6">
          {activeTab === 'overview' && (
            <div className="space-y-6">
              {/* Recent Activity */}
              <div>
                <h3 className="text-lg font-semibold text-black mb-4">Recent Activity</h3>
                <RecentActivityOptimized limit={5} showMarketLinks={false} />
              </div>
            </div>
          )}

          {activeTab === 'transactions' && (
            <TransactionHistory transactions={transactions} />
          )}

          {activeTab === 'settings' && (
            <PaymentMethodsCard 
              paymentMethods={paymentMethods}
              onAddMethod={() => {/* TODO: Implement add payment method */}}
            />
          )}
        </div>
      </div>

      {/* Modals */}
      {showDepositModal && (
        <DepositModal
          profile={profile}
          paymentMethods={paymentMethods}
          onClose={() => setShowDepositModal(false)}
        />
      )}

      {showWithdrawModal && (
        <WithdrawModal
          profile={profile}
          paymentMethods={paymentMethods}
          onClose={() => setShowWithdrawModal(false)}
        />
      )}
    </div>
  )
}

export default memo(WalletDashboardOptimized)
