'use client'

import { useState } from 'react'
import { Profile, PaymentMethod } from '@/lib/types'
import { useToast } from '@/components/ui/Toast'
import { LoadingSpinner, ButtonLoadingSpinner, TransactionLoadingSpinner } from '@/components/ui/LoadingSpinner'

interface WithdrawModalProps {
  profile: Profile
  paymentMethods: PaymentMethod[]
  onClose: () => void
}

export default function WithdrawModal({ profile, paymentMethods, onClose }: WithdrawModalProps) {
  const [amount, setAmount] = useState('')
  const [selectedMethod, setSelectedMethod] = useState<string>('')
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState('')
  const [success, setSuccess] = useState(false)
  const { addToast } = useToast()

  const formatCurrency = (value: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'USD'
    }).format(value)
  }

  const handleWithdraw = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)
    setError('')

    try {
      const withdrawAmount = parseFloat(amount)
      const feeAmount = withdrawAmount * 0.02 // 2% fee
      const totalAmount = withdrawAmount + feeAmount
      
      if (!withdrawAmount || withdrawAmount <= 0) {
        setError('Please enter a valid amount')
        return
      }

      if (withdrawAmount < 25) {
        setError('Minimum withdrawal amount is $25')
        return
      }

      if (totalAmount > profile.balance) {
        setError(`Insufficient balance. Available: ${formatCurrency(profile.balance)}`)
        return
      }

      if (withdrawAmount > profile.daily_withdrawal_limit) {
        setError(`Withdrawal amount exceeds daily limit of ${formatCurrency(profile.daily_withdrawal_limit)}`)
        return
      }

      if (!selectedMethod) {
        setError('Please select a payment method')
        return
      }

      // TODO: Implement actual withdrawal logic with payment processor
      console.log('Withdrawal request:', {
        amount: withdrawAmount,
        feeAmount,
        totalAmount,
        paymentMethodId: selectedMethod,
        userId: profile.id
      })

      // Simulate successful withdrawal request
      setTimeout(() => {
        setLoading(false)
        addToast({
          type: 'success',
          title: 'Withdrawal Requested!',
          message: `Your withdrawal request of ${formatCurrency(withdrawAmount)} has been submitted successfully.`
        })
        setSuccess(true)
        setTimeout(() => {
          setSuccess(false)
          onClose()
        }, 2000)
      }, 2000)

    } catch (err) {
      const errorMessage = 'An error occurred. Please try again.'
      setError(errorMessage)
      addToast({
        type: 'error',
        title: 'Withdrawal Failed',
        message: errorMessage
      })
      setLoading(false)
    }
  }

  const quickAmounts = [100, 250, 500, 1000, profile.balance]

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="bg-white rounded-xl shadow-xl max-w-md w-full mx-4 relative">
        {loading && <LoadingSpinner overlay text="Processing withdrawal..." />}
        <div className="p-6">
          <div className="flex items-center justify-between mb-6">
            <h2 className="text-2xl font-bold text-black">Withdraw Funds</h2>
            <button
              onClick={onClose}
              className="text-black hover:text-black transition-colors"
            >
              <svg className="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
              </svg>
            </button>
          </div>

          <form onSubmit={handleWithdraw} className="space-y-6">
            {/* Available Balance */}
            <div className="bg-green-50 border border-green-200 rounded-lg p-4">
              <div className="flex items-center justify-between">
                <span className="text-sm font-medium text-green-800">Available Balance</span>
                <span className="text-lg font-bold text-green-900">
                  {formatCurrency(profile.balance)}
                </span>
              </div>
            </div>

            {/* Amount Input */}
            <div>
              <label className="block text-sm font-medium text-black mb-2">
                Withdrawal Amount
              </label>
              <div className="relative">
                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                  <span className="text-black sm:text-sm">$</span>
                </div>
                <input
                  type="number"
                  step="0.01"
                  min="25"
                  max={profile.balance}
                  value={amount}
                  onChange={(e) => setAmount(e.target.value)}
                  className="block w-full pl-7 pr-3 py-3 border border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500 !text-black"
                  placeholder="0.00"
                  required
                />
              </div>
              <div className="mt-2 text-sm text-black">
                Daily limit: {formatCurrency(profile.daily_withdrawal_limit)}
              </div>
            </div>

            {/* Quick Amount Buttons */}
            <div>
              <label className="block text-sm font-medium text-black mb-2">
                Quick Amounts
              </label>
              <div className="grid grid-cols-5 gap-2">
                {quickAmounts.map((quickAmount) => (
                  <button
                    key={quickAmount}
                    type="button"
                    onClick={() => setAmount(Math.min(quickAmount, profile.balance).toString())}
                    className="px-3 py-2 text-sm font-medium text-black bg-gray-100 hover:bg-gray-200 rounded-lg transition-colors"
                  >
                    ${quickAmount}
                  </button>
                ))}
              </div>
            </div>

            {/* Payment Method Selection */}
            <div>
              <label className="block text-sm font-medium text-black mb-2">
                Payment Method
              </label>
              {paymentMethods.length === 0 ? (
                <div className="border border-gray-300 rounded-lg p-4 text-center">
                  <p className="text-black mb-3">No payment methods available</p>
                  <button
                    type="button"
                    className="text-blue-600 hover:text-blue-700 font-medium"
                  >
                    Add Payment Method
                  </button>
                </div>
              ) : (
                <select
                  value={selectedMethod}
                  onChange={(e) => setSelectedMethod(e.target.value)}
                  className="block w-full px-3 py-3 border border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500 !text-black"
                  required
                >
                  <option value="">Select a payment method</option>
                  {paymentMethods.map((method) => (
                    <option key={method.id} value={method.id}>
                      {method.type === 'credit_card' ? 'Credit Card' : method.type.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                      {method.last_four && ` •••• ${method.last_four}`}
                      {method.is_default && ' (Default)'}
                    </option>
                  ))}
                </select>
              )}
            </div>

            {/* Fee and Net Amount Calculation */}
            {amount && parseFloat(amount) > 0 && (
              <div className="bg-gray-50 rounded-lg p-4 space-y-2">
                <div className="flex justify-between text-sm">
                  <span className="text-black">Withdrawal Amount:</span>
                  <span className="font-medium">{formatCurrency(parseFloat(amount))}</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-black">Processing Fee (2%):</span>
                  <span className="font-medium text-red-600">-{formatCurrency(parseFloat(amount) * 0.02)}</span>
                </div>
                <div className="border-t border-gray-200 pt-2">
                  <div className="flex justify-between">
                    <span className="font-medium text-black">You'll Receive:</span>
                    <span className="font-bold text-green-600">
                      {formatCurrency(parseFloat(amount) * 0.98)}
                    </span>
                  </div>
                </div>
              </div>
            )}

            {/* Processing Information */}
            <div className="bg-yellow-50 rounded-lg p-4">
              <div className="flex items-center space-x-2">
                <svg className="w-5 h-5 text-yellow-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <span className="text-sm font-medium text-yellow-800">Processing Time</span>
              </div>
              <ul className="mt-2 text-sm text-yellow-700 space-y-1">
                <li>• Withdrawals are processed within 1-3 business days</li>
                <li>• You'll receive an email confirmation</li>
                <li>• 2% processing fee applies</li>
              </ul>
            </div>

            {/* Error Message */}
            {error && (
              <div className="bg-red-50 border border-red-200 rounded-lg p-4">
                <div className="flex">
                  <svg className="w-5 h-5 text-red-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                  </svg>
                  <div className="ml-3">
                    <p className="text-sm text-red-800">{error}</p>
                  </div>
                </div>
              </div>
            )}

            {/* Action Buttons */}
            <div className="flex space-x-3">
              <button
                type="button"
                onClick={onClose}
                className="flex-1 bg-gray-100 hover:bg-gray-200 text-black font-medium py-3 px-4 rounded-lg transition-colors"
              >
                Cancel
              </button>
              <button
                type="submit"
                disabled={loading || !amount || !selectedMethod || parseFloat(amount) < 25}
                className="flex-1 bg-red-600 hover:bg-red-700 disabled:bg-gray-300 text-white font-medium py-3 px-4 rounded-lg transition-colors flex items-center justify-center"
              >
                {loading ? (
                  <ButtonLoadingSpinner text="Processing withdrawal..." />
                ) : (
                  `Withdraw ${formatCurrency(parseFloat(amount) || 0)}`
                )}
              </button>
            </div>
          </form>
        </div>
      </div>
    </div>
  )
}
