import { supabase } from './supabase-client'
import { Market, Bet, Profile, MarketOutcome } from './types'

// Function to record price history
export async function recordPriceHistory(marketId: string, yesPrice: number, noPrice: number, yesPool: number, noPool: number, totalVolume: number) {
  try {
    await supabase
      .from('price_history')
      .insert({
        market_id: marketId,
        yes_price: yesPrice,
        no_price: noPrice,
        yes_pool: yesPool,
        no_pool: noPool,
        total_volume: totalVolume
      })
  } catch (error) {
    // Silently fail if price_history table doesn't exist yet
    console.warn('Failed to record price history:', error)
  }
}

export class BettingEngine {
  /**
   * Calculate liquidity-adjusted price using constant product formula
   * This ensures there's always liquidity available for trading
   */
  static calculateLiquidityAdjustedPrice(
    currentPool: number,
    liquidity: number,
    betAmount: number,
    outcome: string
  ): number {
    // Constant product formula: x * y = k
    // Where x = outcome pool, y = opposite pool, k = constant
    
    const oppositePool = liquidity - currentPool
    
    // Calculate new pools after bet
    const newOutcomePool = currentPool + betAmount
    const newOppositePool = (currentPool * oppositePool) / newOutcomePool
    
    // Calculate price based on new pools
    const totalNewPool = newOutcomePool + newOppositePool
    return newOutcomePool / totalNewPool
  }

  /**
   * Calculate market prices for multiple outcomes with liquidity
   */
  static calculateMultiOutcomePrices(
    outcomes: MarketOutcome[],
    totalLiquidity: number
  ): MarketOutcome[] {
    return outcomes.map(outcome => ({
      ...outcome,
      price: outcome.pool / totalLiquidity
    }))
  }

  /**
   * Add liquidity to a market
   */
  static async addLiquidity(
    userId: string,
    marketId: string,
    amount: number,
    outcomeId?: string // For multi-outcome markets
  ): Promise<{ success: boolean; error?: string; liquidityId?: string }> {
    try {
      // Get market details
      const { data: market, error: marketError } = await supabase
        .from('markets')
        .select('*')
        .eq('id', marketId)
        .single()

      if (marketError || !market) {
        return { success: false, error: 'Market not found' }
      }

      // Check if market is active
      if (market.status !== 'active') {
        return { success: false, error: 'Market is no longer accepting liquidity' }
      }

      // Get user balance
      const { data: profile, error: profileError } = await supabase
        .from('profiles')
        .select('balance')
        .eq('id', userId)
        .single()

      if (profileError || !profile) {
        return { success: false, error: 'User not found' }
      }

      // Check if user has sufficient balance
      if (profile.balance < amount) {
        return { success: false, error: 'Insufficient balance' }
      }

      // Update user balance
      const { error: balanceError } = await supabase
        .from('profiles')
        .update({ balance: profile.balance - amount })
        .eq('id', userId)

      if (balanceError) {
        return { success: false, error: 'Failed to update balance' }
      }

      // Add liquidity to market
      const { error: marketUpdateError } = await supabase
        .from('markets')
        .update({ 
          total_liquidity: (market.total_liquidity || 0) + amount,
          updated_at: new Date().toISOString()
        })
        .eq('id', marketId)

      if (marketUpdateError) {
        // Rollback balance update
        await supabase
          .from('profiles')
          .update({ balance: profile.balance })
          .eq('id', userId)
        return { success: false, error: 'Failed to add liquidity' }
      }

      // Create liquidity transaction record
      const { data: liquidityRecord, error: liquidityError } = await supabase
        .from('liquidity_providers')
        .insert({
          user_id: userId,
          market_id: marketId,
          amount: amount,
          outcome_id: outcomeId,
          created_at: new Date().toISOString()
        })
        .select()
        .single()

      if (liquidityError) {
        return { success: false, error: 'Failed to record liquidity provision' }
      }

      return { success: true, liquidityId: liquidityRecord.id }
    } catch (error) {
      console.error('Error adding liquidity:', error)
      return { success: false, error: 'Internal server error' }
    }
  }

  /**
   * Place bet on multiple outcome market
   */
  static async placeMultiOutcomeBet(
    userId: string,
    marketId: string,
    outcomeId: string,
    amount: number
  ): Promise<{ success: boolean; error?: string; betId?: string }> {
    try {
      // Get market and outcome details
      const { data: market, error: marketError } = await supabase
        .from('markets')
        .select(`
          *,
          outcomes:market_outcomes(*)
        `)
        .eq('id', marketId)
        .single()

      if (marketError || !market) {
        return { success: false, error: 'Market not found' }
      }

      // Check if market is multi-outcome
      if (market.market_type !== 'multi_outcome') {
        return { success: false, error: 'This is not a multi-outcome market' }
      }

      // Find the specific outcome
      const outcome = market.outcomes?.find((o: MarketOutcome) => o.id === outcomeId)
      if (!outcome) {
        return { success: false, error: 'Outcome not found' }
      }

      // Calculate liquidity-adjusted price
      const newPrice = this.calculateLiquidityAdjustedPrice(
        outcome.pool,
        market.total_liquidity || 0,
        amount,
        outcome.name
      )

      // Calculate shares user will receive
      const shares = amount / newPrice

      // Get user balance
      const { data: profile, error: profileError } = await supabase
        .from('profiles')
        .select('balance')
        .eq('id', userId)
        .single()

      if (profileError || !profile) {
        return { success: false, error: 'User not found' }
      }

      // Check if user has sufficient balance
      if (profile.balance < amount) {
        return { success: false, error: 'Insufficient balance' }
      }

      // Update user balance
      const { error: balanceError } = await supabase
        .from('profiles')
        .update({ balance: profile.balance - amount })
        .eq('id', userId)

      if (balanceError) {
        return { success: false, error: 'Failed to update balance' }
      }

      // Update outcome pool
      const { error: outcomeError } = await supabase
        .from('market_outcomes')
        .update({ 
          pool: outcome.pool + amount,
          volume: outcome.volume + amount,
          updated_at: new Date().toISOString()
        })
        .eq('id', outcomeId)

      if (outcomeError) {
        // Rollback balance update
        await supabase
          .from('profiles')
          .update({ balance: profile.balance })
          .eq('id', userId)
        return { success: false, error: 'Failed to update outcome pool' }
      }

      // Create bet record
      const { data: bet, error: betError } = await supabase
        .from('bets')
        .insert({
          user_id: userId,
          market_id: marketId,
          outcome: outcome.name,
          outcome_id: outcomeId,
          amount: amount,
          price: newPrice,
          shares: shares,
          status: 'matched',
          liquidity_contribution: amount,
          created_at: new Date().toISOString()
        })
        .select()
        .single()

      if (betError) {
        return { success: false, error: 'Failed to create bet record' }
      }

      return { success: true, betId: bet.id }
    } catch (error) {
      console.error('Error placing multi-outcome bet:', error)
      return { success: false, error: 'Internal server error' }
    }
  }

  /**
   * Seed a bet (pre-fund before making live)
   */
  static async seedBet(
    userId: string,
    marketId: string,
    outcome: string,
    seededAmount: number,
    price?: number
  ): Promise<{ success: boolean; error?: string; betId?: string }> {
    try {
      // Get market details
      const { data: market, error: marketError } = await supabase
        .from('markets')
        .select('*')
        .eq('id', marketId)
        .single()

      if (marketError || !market) {
        return { success: false, error: 'Market not found' }
      }

      // Check if market is pending (scheduled) - seeding only allowed on scheduled markets
      if (market.status !== 'pending') {
        return { success: false, error: 'Seeding is only allowed on scheduled markets. This market is not in pending status.' }
      }

      // Check if market has ended
      const now = new Date()
      const endDate = new Date(market.end_date)
      if (now >= endDate) {
        return { success: false, error: 'Market has ended' }
      }

      // Check if market already has live volume (seeding only allowed before live trading)
      if ((market.live_volume || 0) > 0) {
        return { success: false, error: 'Market is already live. Seeding is only allowed before live trading begins.' }
      }

      // Get user balance
      const { data: profile, error: profileError } = await supabase
        .from('profiles')
        .select('balance')
        .eq('id', userId)
        .single()

      if (profileError || !profile) {
        return { success: false, error: 'User not found' }
      }

      // Check if user has sufficient balance
      if (profile.balance < seededAmount) {
        return { success: false, error: 'Insufficient balance' }
      }

      // Use current market price if not specified
      const currentPrice = price || (outcome === 'yes' ? market.yes_price : market.no_price)

      // Create seeded bet
      const { data: bet, error: betError } = await supabase
        .from('bets')
        .insert({
          user_id: userId,
          market_id: marketId,
          outcome,
          amount: 0, // Will be updated when bet goes live
          price: currentPrice,
          shares: 0, // Will be calculated when bet goes live
          status: 'seeded',
          seeded_amount: seededAmount,
          live_amount: 0
        })
        .select()
        .single()

      if (betError) {
        return { success: false, error: betError.message }
      }

      // Deduct seeded amount from user balance
      const { error: balanceError } = await supabase
        .from('profiles')
        .update({ balance: profile.balance - seededAmount })
        .eq('id', userId)

      if (balanceError) {
        return { success: false, error: 'Failed to update balance' }
      }

      // Update market seeded volume
      const { error: marketUpdateError } = await supabase
        .from('markets')
        .update({ 
          seeded_volume: (market.seeded_volume || 0) + seededAmount,
          total_volume: (market.total_volume || 0) + seededAmount
        })
        .eq('id', marketId)

      if (marketUpdateError) {
        // Rollback balance update
        await supabase
          .from('profiles')
          .update({ balance: profile.balance })
          .eq('id', userId)
        return { success: false, error: 'Failed to update market volume' }
      }

      // Create transaction record
      await supabase.from('transactions').insert({
        user_id: userId,
        type: 'bet',
        amount: -seededAmount,
        description: `Seeded bet on market (${outcome.toUpperCase()})`,
        market_id: marketId,
        bet_id: bet.id
      })

      return { success: true, betId: bet.id }
    } catch (error: any) {
      return { success: false, error: error.message }
    }
  }

  /**
   * Make a seeded bet live (activate it)
   */
  static async makeBetLive(
    betId: string,
    additionalAmount?: number
  ): Promise<{ success: boolean; error?: string }> {
    try {
      // Get the seeded bet
      const { data: bet, error: betError } = await supabase
        .from('bets')
        .select('*')
        .eq('id', betId)
        .single()

      if (betError || !bet) {
        return { success: false, error: 'Bet not found' }
      }

      if (bet.status !== 'seeded') {
        return { success: false, error: 'Bet is not in seeded status' }
      }

      const totalAmount = bet.seeded_amount + (additionalAmount || 0)
      const liveAmount = additionalAmount || 0

      // If additional amount is provided, check balance
      if (additionalAmount && additionalAmount > 0) {
        const { data: profile, error: profileError } = await supabase
          .from('profiles')
          .select('balance')
          .eq('id', bet.user_id)
          .single()

        if (profileError || !profile) {
          return { success: false, error: 'User profile not found' }
        }

        if (profile.balance < additionalAmount) {
          return { success: false, error: 'Insufficient balance for additional amount' }
        }

        // Deduct additional amount from user balance
        const { error: balanceError } = await supabase
          .from('profiles')
          .update({ balance: profile.balance - additionalAmount })
          .eq('id', bet.user_id)

        if (balanceError) {
          return { success: false, error: 'Failed to update balance' }
        }

        // Create transaction for additional amount
        await supabase.from('transactions').insert({
          user_id: bet.user_id,
          type: 'bet',
          amount: -additionalAmount,
          description: `Additional amount for live bet`,
          market_id: bet.market_id,
          bet_id: bet.id
        })
      }

      // Update bet to live status
      const { error: updateError } = await supabase
        .from('bets')
        .update({
          status: 'live',
          amount: totalAmount,
          shares: totalAmount / bet.price,
          live_amount: liveAmount
        })
        .eq('id', betId)

      if (updateError) {
        return { success: false, error: updateError.message }
      }

      // Update market pools for live bet
      if (liveAmount > 0) {
        const { data: market } = await supabase
          .from('markets')
          .select('*')
          .eq('id', bet.market_id)
          .single()

        if (market) {
          const updateData: any = {
            live_volume: (market.live_volume || 0) + liveAmount
          }

          if (bet.outcome === 'yes') {
            updateData.yes_pool = (market.yes_pool || 0) + liveAmount
          } else {
            updateData.no_pool = (market.no_pool || 0) + liveAmount
          }

          // Calculate new prices
          const totalYesPool = updateData.yes_pool || market.yes_pool || 0
          const totalNoPool = updateData.no_pool || market.no_pool || 0
          const totalPool = totalYesPool + totalNoPool

          if (totalPool > 0) {
            updateData.yes_price = totalYesPool / totalPool
            updateData.no_price = totalNoPool / totalPool
          }

          await supabase
            .from('markets')
            .update(updateData)
            .eq('id', bet.market_id)
        }
      }

      // Try to match with existing bets
      await this.matchBets(bet.market_id)

      return { success: true }
    } catch (error: any) {
      return { success: false, error: error.message }
    }
  }

  /**
   * Check if market shares are exhausted
   */
  static async checkShareExhaustion(marketId: string): Promise<{ exhausted: boolean; reason?: string }> {
    try {
      const { data: market, error } = await supabase
        .from('markets')
        .select('total_shares, shares_sold, status, market_type, outcomes')
        .eq('id', marketId)
        .single()

      if (error || !market) {
        return { exhausted: false, reason: 'Market not found' }
      }

      // Check if market is already closed/resolved
      if (['closed', 'resolved', 'refunded'].includes(market.status)) {
        return { exhausted: true, reason: 'Market is closed' }
      }

      // For binary markets
      if (market.market_type === 'binary') {
        const totalShares = market.total_shares || 1000
        const sharesSold = market.shares_sold || 0
        const remainingShares = totalShares - sharesSold

        if (remainingShares <= 0) {
          return { exhausted: true, reason: 'All shares have been sold' }
        }
      }

      // For multi-outcome markets
      if (market.market_type === 'multi_outcome' && market.outcomes) {
        for (const outcome of market.outcomes) {
          const outcomeShares = outcome.pool || 0
          const maxSharesPerOutcome = (market.total_shares || 1000) / market.outcomes.length
          
          if (outcomeShares >= maxSharesPerOutcome) {
            return { exhausted: true, reason: `Outcome "${outcome.name}" shares exhausted` }
          }
        }
      }

      return { exhausted: false }
    } catch (error) {
      console.error('Error checking share exhaustion:', error)
      return { exhausted: false, reason: 'Error checking shares' }
    }
  }

  /**
   * Update market status when shares are exhausted
   */
  static async updateMarketForExhaustion(marketId: string): Promise<boolean> {
    try {
      const { error } = await supabase
        .from('markets')
        .update({
          status: 'closed',
          admin_notes: 'Market closed - shares exhausted',
          updated_at: new Date().toISOString()
        })
        .eq('id', marketId)

      if (error) {
        console.error('Failed to update market for exhaustion:', error)
        return false
      }

      // Send notification to market creator
      try {
        const { data: market } = await supabase
          .from('markets')
          .select('creator_id, title')
          .eq('id', marketId)
          .single()

        if (market) {
          const { NotificationService } = await import('@/lib/services/notificationService')
          await NotificationService.createEventNotification(
            market.creator_id,
            'market_shares_exhausted',
            'Market Shares Exhausted',
            `Your market "${market.title}" has been closed due to share exhaustion.`,
            {
              market: {
                id: marketId,
                title: market.title,
                reason: 'shares_exhausted'
              }
            },
            true // Send email
          )
        }
      } catch (notificationError) {
        console.error('Failed to send exhaustion notification:', notificationError)
      }

      return true
    } catch (error) {
      console.error('Error updating market for exhaustion:', error)
      return false
    }
  }

  /**
   * Place a share-based bet on a market
   */
  static async placeBet(
    userId: string,
    marketId: string,
    outcome: 'yes' | 'no',
    amount: number,
    shares?: number
  ): Promise<{ success: boolean; error?: string; betId?: string; sharesPurchased?: number; sharePrice?: number; potentialPayout?: number }> {
    try {
      // Check if shares are exhausted before placing bet
      const exhaustionCheck = await this.checkShareExhaustion(marketId)
      if (exhaustionCheck.exhausted) {
        return { 
          success: false, 
          error: `Cannot place bet: ${exhaustionCheck.reason}` 
        }
      }

      // Get market details
      const { data: market, error: marketError } = await supabase
        .from('markets')
        .select('*')
        .eq('id', marketId)
        .single()

      if (marketError || !market) {
        return { success: false, error: 'Market not found' }
      }

      // Check if market is active
      if (market.status !== 'active') {
        return { success: false, error: 'Market is no longer accepting bets' }
      }

      // Check if market has ended
      const now = new Date()
      const endDate = new Date(market.end_date)
      if (now >= endDate) {
        return { success: false, error: 'Market has ended' }
      }

      // Note: shares_remaining column has been removed - prediction markets typically have unlimited shares

      // Get user balance
      const { data: profile, error: profileError } = await supabase
        .from('profiles')
        .select('balance')
        .eq('id', userId)
        .single()

      if (profileError || !profile) {
        return { success: false, error: 'User not found' }
      }

      // Check if user has sufficient balance
      if (profile.balance < amount) {
        return { success: false, error: 'Insufficient balance' }
      }

      // Try the new share-based betting function first
      try {
        const { data: result, error: betError } = await supabase
          .rpc('place_share_bet', {
            p_user_id: userId,
            p_market_id: marketId,
            p_outcome: outcome,
            p_amount: amount,
            p_shares: shares || null
          })
          .single()

        if (betError) {
          throw betError
        }

        const betResult = result as any
        if (!betResult.success) {
          throw new Error(betResult.message)
        }

        // Return success with share information
        return {
          success: true,
          betId: betResult.bet_id,
          sharesPurchased: betResult.shares_purchased,
          sharePrice: betResult.share_price,
          potentialPayout: betResult.potential_payout
        }
      } catch (shareBetError) {
        // Fallback to old betting system if share-based functions don't exist
        console.warn('Share-based betting not available, falling back to old system:', shareBetError)
        
        // Use current market price if not specified
        const betPrice = market.yes_price || market.no_price || 0.5
        const calculatedShares = shares || (amount / betPrice)
        
        // Check if purchase would exceed total shares available
        const currentSharesSold = market.shares_sold || 0
        const totalShares = market.total_shares || 1000
        if (currentSharesSold + calculatedShares > totalShares) {
          const remainingShares = totalShares - currentSharesSold
          throw new Error(`Purchase would exceed total shares available. Only ${remainingShares.toFixed(2)} shares remaining.`)
        }

        // Create live bet using old system
        const { data: bet, error: betError } = await supabase
          .from('bets')
          .insert({
            user_id: userId,
            market_id: marketId,
            outcome,
            amount,
            price: betPrice,
            shares: calculatedShares,
            status: 'matched',
            seeded_amount: 0,
            live_amount: amount
          })
          .select()
          .single()

        if (betError) {
          throw new Error(betError.message)
        }

        // Deduct amount from user balance
        const { error: balanceError } = await supabase
          .from('profiles')
          .update({ balance: profile.balance - amount })
          .eq('id', userId)

        if (balanceError) {
          throw new Error('Failed to update balance')
        }

        // Update market pools, volumes, and shares sold
        const updateData: any = {
          live_volume: (market.live_volume || 0) + amount,
          total_volume: (market.total_volume || 0) + amount,
          shares_sold: (market.shares_sold || 0) + calculatedShares
        }

        if (outcome === 'yes') {
          updateData.yes_pool = (market.yes_pool || 0) + amount
          updateData.current_yes_share_price = betPrice
        } else {
          updateData.no_pool = (market.no_pool || 0) + amount
          updateData.current_no_share_price = betPrice
        }

        // Calculate new prices based on updated pools
        const totalYesPool = updateData.yes_pool || market.yes_pool || 0
        const totalNoPool = updateData.no_pool || market.no_pool || 0
        const totalPool = totalYesPool + totalNoPool

        if (totalPool > 0) {
          updateData.yes_price = totalYesPool / totalPool
          updateData.no_price = totalNoPool / totalPool
        }

        const { error: marketUpdateError } = await supabase
          .from('markets')
          .update(updateData)
          .eq('id', marketId)

        if (marketUpdateError) {
          // Rollback balance update
          await supabase
            .from('profiles')
            .update({ balance: profile.balance })
            .eq('id', userId)
          throw new Error('Failed to update market pools')
        }

        // Create unified transaction record
        await supabase.from('unified_transactions').insert({
          user_id: userId,
          type: 'bet',
          status: 'completed',
          amount: -amount,
          currency: 'USD',
          description: `Live bet placed on market (${outcome.toUpperCase()})`,
          market_id: marketId,
          bet_id: bet.id,
          fee_amount: 0,
          net_amount: -amount,
          processed_at: new Date().toISOString(),
          metadata: {
            outcome: outcome,
            price: betPrice,
            shares: calculatedShares
          }
        })

        // Check if shares are now exhausted after this bet
        const postBetExhaustion = await this.checkShareExhaustion(marketId)
        if (postBetExhaustion.exhausted) {
          console.log('Shares exhausted after bet, updating market status')
          await this.updateMarketForExhaustion(marketId)
        }

        return { 
          success: true, 
          betId: bet.id,
          sharesPurchased: calculatedShares,
          sharePrice: betPrice,
          potentialPayout: calculatedShares * 1.00 // Estimate $1 per share
        }
      }
    } catch (error: any) {
      return { success: false, error: error.message }
    }
  }

  /**
   * Match pending bets for a market
   */
  static async matchBets(marketId: string): Promise<void> {
    try {
      // Get all pending bets for this market
      const { data: pendingBets, error: fetchError } = await supabase
        .from('bets')
        .select('*')
        .eq('market_id', marketId)
        .eq('status', 'live')
        .order('created_at', { ascending: true })

      if (fetchError) {
        console.error('Error fetching pending bets:', fetchError)
        return
      }

      if (!pendingBets || pendingBets.length === 0) {
        return
      }

      // Simple matching logic - match opposite outcome bets
      const yesBets = pendingBets.filter(bet => bet.outcome === 'yes')
      const noBets = pendingBets.filter(bet => bet.outcome === 'no')

      // Match bets with similar amounts
      const minMatches = Math.min(yesBets.length, noBets.length)
      
      for (let i = 0; i < minMatches; i++) {
        const yesBet = yesBets[i]
        const noBet = noBets[i]
        
        // Update both bets to matched status
        await supabase
          .from('bets')
          .update({ status: 'matched' })
          .eq('id', yesBet.id)
          
        await supabase
          .from('bets')
          .update({ status: 'matched' })
          .eq('id', noBet.id)
      }

    } catch (error) {
      console.error('Error matching bets:', error)
    }
  }

  /**
   * Automatically determine the winning outcome based on market data
   * This can be extended with various algorithms (price-based, volume-based, etc.)
   */
  static async determineWinningOutcome(
    marketId: string
  ): Promise<{ outcome: 'yes' | 'no' | null; confidence: number; reasoning: string }> {
    try {
      // Get market data
      const { data: market, error: marketError } = await supabase
        .from('markets')
        .select(`
          id,
          title,
          description,
          category,
          end_date,
          yes_price,
          no_price,
          total_volume,
          total_liquidity,
          status
        `)
        .eq('id', marketId)
        .single()

      if (marketError || !market) {
        return { outcome: null, confidence: 0, reasoning: 'Market not found' }
      }

      // Get bet data for analysis
      const { data: bets, error: betsError } = await supabase
        .from('bets')
        .select('outcome, amount, seeded_amount, live_amount, status')
        .eq('market_id', marketId)
        .eq('status', 'matched')

      if (betsError) {
        return { outcome: null, confidence: 0, reasoning: 'Failed to fetch bet data' }
      }

      if (!bets || bets.length === 0) {
        return { outcome: null, confidence: 0, reasoning: 'No matched bets found' }
      }

      // Calculate metrics for each outcome
      const yesBets = bets.filter(bet => bet.outcome === 'yes')
      const noBets = bets.filter(bet => bet.outcome === 'no')

      const yesMetrics = {
        count: yesBets.length,
        totalAmount: yesBets.reduce((sum, bet) => sum + (bet.amount || 0), 0),
        seededAmount: yesBets.reduce((sum, bet) => sum + (bet.seeded_amount || 0), 0),
        liveAmount: yesBets.reduce((sum, bet) => sum + (bet.live_amount || 0), 0),
        avgAmount: yesBets.length > 0 ? yesBets.reduce((sum, bet) => sum + (bet.amount || 0), 0) / yesBets.length : 0
      }

      const noMetrics = {
        count: noBets.length,
        totalAmount: noBets.reduce((sum, bet) => sum + (bet.amount || 0), 0),
        seededAmount: noBets.reduce((sum, bet) => sum + (bet.seeded_amount || 0), 0),
        liveAmount: noBets.reduce((sum, bet) => sum + (bet.live_amount || 0), 0),
        avgAmount: noBets.length > 0 ? noBets.reduce((sum, bet) => sum + (bet.amount || 0), 0) / noBets.length : 0
      }

      // Algorithm 1: Price-based determination (most reliable for prediction markets)
      let priceBasedOutcome: 'yes' | 'no' | null = null
      let priceConfidence = 0
      
      if (market.yes_price && market.no_price) {
        const priceDiff = Math.abs(market.yes_price - market.no_price)
        const threshold = 0.1 // 10% price difference threshold for high confidence
        
        if (market.yes_price > market.no_price + threshold) {
          priceBasedOutcome = 'yes'
          priceConfidence = Math.min(priceDiff * 100, 95) // Cap at 95% confidence
        } else if (market.no_price > market.yes_price + threshold) {
          priceBasedOutcome = 'no'
          priceConfidence = Math.min(priceDiff * 100, 95)
        } else {
          priceConfidence = 50 // Low confidence when prices are close
        }
      }

      // Algorithm 2: Volume-based determination
      let volumeBasedOutcome: 'yes' | 'no' | null = null
      let volumeConfidence = 0
      
      const totalVolume = yesMetrics.totalAmount + noMetrics.totalAmount
      if (totalVolume > 0) {
        const yesVolumeRatio = yesMetrics.totalAmount / totalVolume
        const noVolumeRatio = noMetrics.totalAmount / totalVolume
        const volumeDiff = Math.abs(yesVolumeRatio - noVolumeRatio)
        
        if (yesVolumeRatio > noVolumeRatio + 0.2) { // 20% difference threshold
          volumeBasedOutcome = 'yes'
          volumeConfidence = Math.min(volumeDiff * 100, 80)
        } else if (noVolumeRatio > yesVolumeRatio + 0.2) {
          volumeBasedOutcome = 'no'
          volumeConfidence = Math.min(volumeDiff * 100, 80)
        } else {
          volumeConfidence = 50
        }
      }

      // Algorithm 3: Seeded amount analysis (early money)
      let seededBasedOutcome: 'yes' | 'no' | null = null
      let seededConfidence = 0
      
      const totalSeeded = yesMetrics.seededAmount + noMetrics.seededAmount
      if (totalSeeded > 0) {
        const yesSeededRatio = yesMetrics.seededAmount / totalSeeded
        const noSeededRatio = noMetrics.seededAmount / totalSeeded
        const seededDiff = Math.abs(yesSeededRatio - noSeededRatio)
        
        if (yesSeededRatio > noSeededRatio + 0.3) { // 30% difference threshold for seeded
          seededBasedOutcome = 'yes'
          seededConfidence = Math.min(seededDiff * 100, 70)
        } else if (noSeededRatio > yesSeededRatio + 0.3) {
          seededBasedOutcome = 'no'
          seededConfidence = Math.min(seededDiff * 100, 70)
        } else {
          seededConfidence = 50
        }
      }

      // Combine algorithms with weighted scoring
      const algorithms = [
        { outcome: priceBasedOutcome, confidence: priceConfidence, weight: 0.5, name: 'price' },
        { outcome: volumeBasedOutcome, confidence: volumeConfidence, weight: 0.3, name: 'volume' },
        { outcome: seededBasedOutcome, confidence: seededConfidence, weight: 0.2, name: 'seeded' }
      ]

      // Calculate weighted scores for each outcome
      const yesScore = algorithms.reduce((score, algo) => {
        return score + (algo.outcome === 'yes' ? algo.confidence * algo.weight : 0)
      }, 0)

      const noScore = algorithms.reduce((score, algo) => {
        return score + (algo.outcome === 'no' ? algo.confidence * algo.weight : 0)
      }, 0)

      // Determine final outcome
      let finalOutcome: 'yes' | 'no' | null = null
      let finalConfidence = 0
      let reasoning = ''

      if (yesScore > noScore + 10) { // 10 point difference threshold
        finalOutcome = 'yes'
        finalConfidence = Math.min(yesScore, 95)
        reasoning = `YES selected based on: ${algorithms.filter(a => a.outcome === 'yes').map(a => `${a.name}(${a.confidence.toFixed(1)}%)`).join(', ')}`
      } else if (noScore > yesScore + 10) {
        finalOutcome = 'no'
        finalConfidence = Math.min(noScore, 95)
        reasoning = `NO selected based on: ${algorithms.filter(a => a.outcome === 'no').map(a => `${a.name}(${a.confidence.toFixed(1)}%)`).join(', ')}`
      } else {
        finalOutcome = null
        finalConfidence = Math.max(yesScore, noScore)
        reasoning = `Insufficient confidence for automatic resolution. YES: ${yesScore.toFixed(1)}%, NO: ${noScore.toFixed(1)}%`
      }

      return {
        outcome: finalOutcome,
        confidence: finalConfidence,
        reasoning: reasoning
      }

    } catch (error: any) {
      console.error('Error determining winning outcome:', error)
      return { 
        outcome: null, 
        confidence: 0, 
        reasoning: `Error in automatic determination: ${error.message}` 
      }
    }
  }

  /**
   * Resolve a market with comprehensive payout system
   * Uses the new payout system to handle all stakeholders: winners, creators, and admin
   * Can automatically determine winners if outcome is not provided
   */
  static async resolveMarket(
    marketId: string,
    outcome?: string
  ): Promise<{ success: boolean; error?: string; payoutSummary?: any; autoDetermined?: boolean; confidence?: number }> {
    try {
      // Check if market exists and has bets
      const { data: market, error: marketError } = await supabase
        .from('markets')
        .select(`
          id, 
          status, 
          market_type,
          outcomes:market_outcomes (
            id,
            name,
            price,
            pool
          )
        `)
        .eq('id', marketId)
        .single()

      if (marketError || !market) {
        return { success: false, error: 'Market not found' }
      }

      if (market.status !== 'active' && market.status !== 'closed') {
        return { success: false, error: 'Market is not in a resolvable state' }
      }

      // If no outcome provided, try to determine it automatically
      let finalOutcome = outcome
      let autoDetermined = false
      let confidence = 0

      if (!outcome) {
        const determination = await this.determineWinningOutcome(marketId)
        
        if (determination.outcome) {
          finalOutcome = determination.outcome
          autoDetermined = true
          confidence = determination.confidence
          console.log(`Auto-determined outcome: ${determination.outcome} (${determination.confidence.toFixed(1)}% confidence)`)
          console.log(`Reasoning: ${determination.reasoning}`)
        } else {
          return { 
            success: false, 
            error: `Cannot automatically determine winner: ${determination.reasoning}`,
            confidence: determination.confidence
          }
        }
      } else {
        // Validate the provided outcome
        if (market.market_type === 'binary') {
          if (outcome !== 'yes' && outcome !== 'no') {
            return { success: false, error: 'Invalid outcome for binary market. Must be "yes" or "no".' }
          }
        } else if (market.market_type === 'multi_outcome') {
          const validOutcomes = market.outcomes?.map(o => o.name) || []
          if (!validOutcomes.includes(outcome)) {
            return { 
              success: false, 
              error: `Invalid outcome for multi-outcome market. Valid outcomes: ${validOutcomes.join(', ')}` 
            }
          }
        }
      }

      // Check if there are any matched bets
      const { data: bets, error: betsError } = await supabase
        .from('bets')
        .select('id')
        .eq('market_id', marketId)
        .eq('status', 'matched')
        .limit(1)

      if (betsError) {
        return { success: false, error: betsError.message }
      }

      // If no bets, just update market status
      if (!bets || bets.length === 0) {
        const { error: updateError } = await supabase
          .from('markets')
          .update({ 
            status: 'resolved', 
            outcome: finalOutcome, 
            resolution_date: new Date().toISOString() 
          })
          .eq('id', marketId)

        if (updateError) {
          return { success: false, error: updateError.message }
        }
        return { 
          success: true, 
          payoutSummary: { totalPayouts: 0, totalAmount: 0 },
          autoDetermined,
          confidence
        }
      }

      // Try to use the comprehensive payout system, fallback to basic resolution if not available
      let payoutProcessed = false
      let payoutSummary = null

      try {
        const { data: payoutResult, error: payoutError } = await supabase
          .rpc('process_market_payouts', {
            p_market_id: marketId,
            p_outcome: finalOutcome
          })

        if (payoutError) {
          console.error('Error processing payouts:', payoutError)
          
          // Check if it's because the payout system isn't set up
          if (payoutError.message && (
              payoutError.message.includes('function process_market_payouts') || 
              payoutError.message.includes('does not exist'))) {
            console.log('Payout system not set up, using basic resolution')
          } else {
            console.error('Payout error:', payoutError.message || 'Unknown payout error')
          }
        } else {
          const result = payoutResult?.[0]
          if (result && result.success) {
            payoutProcessed = true
            payoutSummary = result
          }
        }
      } catch (rpcError) {
        console.error('RPC call failed:', rpcError)
      }

      // If payout system failed, do basic market resolution
      if (!payoutProcessed) {
        console.log('Using basic market resolution')
        
        // Update market status
        const { error: updateError } = await supabase
          .from('markets')
          .update({
            status: 'resolved',
            outcome: finalOutcome,
            resolved_at: new Date().toISOString()
          })
          .eq('id', marketId)

        if (updateError) {
          console.error('Error updating market status:', updateError)
          return { success: false, error: 'Failed to update market status' }
        }

        payoutSummary = {
          total_payouts: 0,
          total_amount: 0,
          winner_count: 0,
          message: 'Market resolved (basic resolution - payout system not available)'
        }
      }

      // Update bet statuses to reflect resolution
      // First get all matched bets for this market
      const { data: matchedBets, error: fetchError } = await supabase
        .from('bets')
        .select('id, outcome')
        .eq('market_id', marketId)
        .eq('status', 'matched')

      if (fetchError) {
        console.error('Error fetching matched bets:', fetchError)
      } else if (matchedBets) {
        // Update each bet individually based on outcome
        for (const bet of matchedBets) {
          const newStatus = bet.outcome === finalOutcome ? 'won' : 'lost'
          await supabase
            .from('bets')
            .update({ status: newStatus })
            .eq('id', bet.id)
        }
      }

      return { 
        success: true, 
        payoutSummary: payoutSummary ? {
          totalPayouts: payoutSummary.total_payouts || 0,
          totalAmount: payoutSummary.total_amount || 0,
          message: payoutSummary.message || 'Market resolved successfully'
        } : {
          totalPayouts: 0,
          totalAmount: 0,
          message: 'Market resolved (basic resolution)'
        },
        autoDetermined,
        confidence
      }
    } catch (error: any) {
      console.error('Error in resolveMarket:', error)
      return { success: false, error: error.message }
    }
  }

  /**
   * Automatically resolve markets that have ended and have sufficient confidence
   * This method can be called by a cron job or scheduled task
   */
  static async autoResolveExpiredMarkets(
    minConfidence: number = 70
  ): Promise<{ 
    success: boolean; 
    error?: string; 
    processed: number; 
    resolved: number; 
    failed: number;
    results: Array<{
      marketId: string;
      marketTitle: string;
      outcome: 'yes' | 'no' | null;
      confidence: number;
      reasoning: string;
      success: boolean;
      error?: string;
    }>
  }> {
    try {
      console.log('Starting automatic market resolution...')
      
      // Get all expired markets that are still active or closed
      const { data: expiredMarkets, error: marketsError } = await supabase
        .from('markets')
        .select('id, title, end_date, status')
        .in('status', ['active', 'closed'])
        .lt('end_date', new Date().toISOString())
        .order('end_date', { ascending: true })

      if (marketsError) {
        return {
          success: false,
          error: marketsError.message,
          processed: 0,
          resolved: 0,
          failed: 0,
          results: []
        }
      }

      if (!expiredMarkets || expiredMarkets.length === 0) {
        console.log('No expired markets found for auto-resolution')
        return {
          success: true,
          processed: 0,
          resolved: 0,
          failed: 0,
          results: []
        }
      }

      console.log(`Found ${expiredMarkets.length} expired markets to process`)

      const results: Array<{
        marketId: string;
        marketTitle: string;
        outcome: 'yes' | 'no' | null;
        confidence: number;
        reasoning: string;
        success: boolean;
        error?: string;
      }> = []

      let resolvedCount = 0
      let failedCount = 0

      // Process each expired market
      for (const market of expiredMarkets) {
        console.log(`Processing market: ${market.title} (${market.id})`)
        
        try {
          // Determine the winning outcome
          const determination = await this.determineWinningOutcome(market.id)
          
          if (determination.outcome && determination.confidence >= minConfidence) {
            // Auto-resolve the market
            const resolveResult = await this.resolveMarket(market.id, determination.outcome)
            
            if (resolveResult.success) {
              resolvedCount++
              console.log(`✅ Auto-resolved market "${market.title}": ${determination.outcome} (${determination.confidence.toFixed(1)}% confidence)`)
              
              results.push({
                marketId: market.id,
                marketTitle: market.title,
                outcome: determination.outcome,
                confidence: determination.confidence,
                reasoning: determination.reasoning,
                success: true
              })
            } else {
              failedCount++
              console.log(`❌ Failed to resolve market "${market.title}": ${resolveResult.error}`)
              
              results.push({
                marketId: market.id,
                marketTitle: market.title,
                outcome: determination.outcome,
                confidence: determination.confidence,
                reasoning: determination.reasoning,
                success: false,
                error: resolveResult.error
              })
            }
          } else {
            // Insufficient confidence for auto-resolution
            console.log(`⚠️ Insufficient confidence for market "${market.title}": ${determination.confidence.toFixed(1)}% (min: ${minConfidence}%)`)
            
            results.push({
              marketId: market.id,
              marketTitle: market.title,
              outcome: determination.outcome,
              confidence: determination.confidence,
              reasoning: determination.reasoning,
              success: false,
              error: `Insufficient confidence: ${determination.confidence.toFixed(1)}% < ${minConfidence}%`
            })
          }
        } catch (error: any) {
          failedCount++
          console.error(`Error processing market "${market.title}":`, error)
          
          results.push({
            marketId: market.id,
            marketTitle: market.title,
            outcome: null,
            confidence: 0,
            reasoning: 'Error during processing',
            success: false,
            error: error.message
          })
        }
      }

      console.log(`Auto-resolution completed: ${resolvedCount} resolved, ${failedCount} failed`)
      
      return {
        success: true,
        processed: expiredMarkets.length,
        resolved: resolvedCount,
        failed: failedCount,
        results
      }

    } catch (error: any) {
      console.error('Error in autoResolveExpiredMarkets:', error)
      return {
        success: false,
        error: error.message,
        processed: 0,
        resolved: 0,
        failed: 0,
        results: []
      }
    }
  }

  /**
   * Cancel a market and process refunds
   */
  static async cancelMarket(
    marketId: string,
    reason: string = 'Market cancelled by admin'
  ): Promise<{ success: boolean; error?: string; refundSummary?: any }> {
    try {
      // Try to use the refund processing system, fallback to basic cancellation if not available
      let refundProcessed = false
      let refundSummary = null

      try {
        const { data: refundResult, error: refundError } = await supabase
          .rpc('process_market_refunds', {
            p_market_id: marketId
          })

        if (refundError) {
          console.error('Error processing refunds:', refundError)
          
          // Check if it's because the refund system isn't set up
          if (refundError.message && (
              refundError.message.includes('function process_market_refunds') || 
              refundError.message.includes('does not exist'))) {
            console.log('Refund system not set up, using basic cancellation')
          } else {
            console.error('Refund error:', refundError.message || 'Unknown refund error')
          }
        } else {
          const result = refundResult?.[0]
          if (result && result.success) {
            refundProcessed = true
            refundSummary = result
          }
        }
      } catch (rpcError) {
        console.error('Refund RPC call failed:', rpcError)
      }

      // If refund system failed, do basic market cancellation
      if (!refundProcessed) {
        console.log('Using basic market cancellation')
        
        // Update market status to cancelled
        const { error: updateError } = await supabase
          .from('markets')
          .update({
            status: 'cancelled',
            cancelled_at: new Date().toISOString(),
            cancellation_reason: reason
          })
          .eq('id', marketId)

        if (updateError) {
          console.error('Error updating market status:', updateError)
          return { success: false, error: 'Failed to update market status' }
        }

        refundSummary = {
          total_refunds: 0,
          refunded_bets: 0,
          message: 'Market cancelled (basic cancellation - refund system not available)'
        }
      }

      // Update bet statuses to cancelled
      const { error: betUpdateError } = await supabase
        .from('bets')
        .update({ status: 'cancelled' })
        .eq('market_id', marketId)
        .in('status', ['seeded', 'live', 'matched'])

      if (betUpdateError) {
        console.error('Error updating bet statuses:', betUpdateError)
        // Don't fail the entire operation for this
      }

      return { 
        success: true, 
        refundSummary: refundSummary ? {
          totalRefunds: refundSummary.total_refunds || 0,
          totalAmount: refundSummary.total_amount || 0,
          message: refundSummary.message || 'Market cancelled successfully'
        } : {
          totalRefunds: 0,
          totalAmount: 0,
          message: 'Market cancelled (basic cancellation)'
        }
      }
    } catch (error: any) {
      console.error('Error in cancelMarket:', error)
      return { success: false, error: error.message }
    }
  }

  /**
   * Calculate market prices based on seeded and live betting system
   * Prices reflect the distribution of both seeded and live money
   */
  static async calculateMarketPrices(marketId: string): Promise<{
    yes_price: number
    no_price: number
    total_volume: number
    yes_pool: number
    no_pool: number
    seeded_volume: number
    live_volume: number
  }> {
    try {
      // Get all matched bets for this market
      const { data: bets, error } = await supabase
        .from('bets')
        .select('seeded_amount, live_amount, outcome')
        .eq('market_id', marketId)
        .eq('status', 'matched')

      if (error) {
        console.error('Error fetching bets:', error)
        return { 
          yes_price: 0.5, 
          no_price: 0.5, 
          total_volume: 0, 
          yes_pool: 0, 
          no_pool: 0,
          seeded_volume: 0,
          live_volume: 0
        }
      }

      // Calculate pools for each outcome and type
      let yesSeededPool = 0
      let noSeededPool = 0
      let yesLivePool = 0
      let noLivePool = 0
      let totalSeededVolume = 0
      let totalLiveVolume = 0

      bets?.forEach((bet: any) => {
        const seededAmount = bet.seeded_amount || 0
        const liveAmount = bet.live_amount || 0
        
        totalSeededVolume += seededAmount
        totalLiveVolume += liveAmount
        
        if (bet.outcome === 'yes') {
          yesSeededPool += seededAmount
          yesLivePool += liveAmount
        } else {
          noSeededPool += seededAmount
          noLivePool += liveAmount
        }
      })

      const totalVolume = totalSeededVolume + totalLiveVolume
      const yesPool = yesSeededPool + yesLivePool
      const noPool = noSeededPool + noLivePool

      // Calculate prices based on total pool sizes
      let yesPrice = 0.5 // Default to 50/50 if no bets
      let noPrice = 0.5

      if (totalVolume > 0) {
        yesPrice = yesPool / totalVolume
        noPrice = noPool / totalVolume
      }

      return {
        yes_price: yesPrice,
        no_price: noPrice,
        total_volume: totalVolume,
        yes_pool: yesPool,
        no_pool: noPool,
        seeded_volume: totalSeededVolume,
        live_volume: totalLiveVolume
      }
    } catch (error) {
      console.error('Error calculating market prices:', error)
      return { 
        yes_price: 0.5, 
        no_price: 0.5, 
        total_volume: 0, 
        yes_pool: 0, 
        no_pool: 0,
        seeded_volume: 0,
        live_volume: 0
      }
    }
  }

  /**
   * Get user's portfolio
   */
  static async getUserPortfolio(userId: string): Promise<{
    profile: Profile | null
    activeBets: Bet[]
    totalWinnings: number
    totalBets: number
  }> {
    try {
      // Get user profile
      const { data: profile } = await supabase
        .from('profiles')
        .select('*')
        .eq('id', userId)
        .single()

      // Get user's active bets
      const { data: activeBets } = await supabase
        .from('bets')
        .select(`
          *,
          markets (
            title,
            status,
            outcome
          )
        `)
        .eq('user_id', userId)
        .eq('status', 'matched')

      // Get user's transaction history for winnings calculation
      const { data: transactions } = await supabase
        .from('transactions')
        .select('*')
        .eq('user_id', userId)
        .eq('type', 'win')

      const totalWinnings = transactions?.reduce((sum: number, tx: any) => sum + tx.amount, 0) || 0
      const totalBets = activeBets?.length || 0

      return {
        profile,
        activeBets: activeBets || [],
        totalWinnings,
        totalBets
      }
    } catch (error) {
      console.error('Error getting user portfolio:', error)
      return {
        profile: null,
        activeBets: [],
        totalWinnings: 0,
        totalBets: 0
      }
    }
  }

  /**
   * Get market analytics
   */
  static async getMarketAnalytics(marketId: string): Promise<{
    totalBets: number
    totalVolume: number
    yesBets: number
    noBets: number
    uniqueUsers: number
    priceHistory: Array<{ timestamp: string; yes_price: number; no_price: number }>
  }> {
    try {
      // Get all bets for this market
      const { data: bets } = await supabase
        .from('bets')
        .select('*')
        .eq('market_id', marketId)
        .eq('status', 'matched')

      const totalBets = bets?.length || 0
      const totalVolume = bets?.reduce((sum: number, bet: any) => sum + bet.amount, 0) || 0
      const yesBets = bets?.filter((bet: any) => bet.outcome === 'yes').length || 0
      const noBets = bets?.filter((bet: any) => bet.outcome === 'no').length || 0
      
      // Get unique users
      const uniqueUsers = new Set(bets?.map((bet: any) => bet.user_id)).size

      // No mock data - return empty price history
      const priceHistory: any[] = []

      return {
        totalBets,
        totalVolume,
        yesBets,
        noBets,
        uniqueUsers,
        priceHistory
      }
    } catch (error) {
      console.error('Error getting market analytics:', error)
      return {
        totalBets: 0,
        totalVolume: 0,
        yesBets: 0,
        noBets: 0,
        uniqueUsers: 0,
        priceHistory: []
      }
    }
  }


  /**
   * Cancel a pending bet
   */
  static async cancelBet(betId: string, userId: string): Promise<{ success: boolean; error?: string }> {
    try {
      // Check if bet exists and belongs to user
      const { data: bet, error: betError } = await supabase
        .from('bets')
        .select('*')
        .eq('id', betId)
        .eq('user_id', userId)
        .single()

      if (betError || !bet) {
        return { success: false, error: 'Bet not found' }
      }

      // Check if bet is pending
      if (bet.status !== 'pending') {
        return { success: false, error: 'Cannot cancel matched bet' }
      }

      // Update bet status
      const { error: updateError } = await supabase
        .from('bets')
        .update({ status: 'cancelled' })
        .eq('id', betId)

      if (updateError) {
        return { success: false, error: updateError.message }
      }

      // Refund the user
      const { error: refundError } = await supabase
        .from('profiles')
        .update({ balance: bet.amount })
        .eq('id', userId)

      if (refundError) {
        return { success: false, error: refundError.message }
      }

      return { success: true }
    } catch (error: any) {
      return { success: false, error: error.message }
    }
  }
}
