import { useState, useEffect, useCallback, useMemo } from 'react'
import { supabase } from '@/lib/supabase-client'

export interface AdminBet {
  id: string
  user_id: string
  market_id: string
  outcome: string
  amount: number
  price: number
  shares: number
  status: string
  seeded_amount: number
  live_amount: number
  potential_payout: number
  created_at: string
  updated_at: string
  user_profile?: {
    username: string
    email: string
  }
  market?: {
    title: string
    status: string
    yes_price: number
    no_price: number
    total_liquidity: number
  }
  algorithm_metrics?: {
    price_impact: number
    liquidity_utilization: number
    market_efficiency: number
    risk_score: number
  }
}

export interface BetMetrics {
  totalBets: number
  totalVolume: number
  seededVolume: number
  liveVolume: number
  matchedBets: number
  pendingBets: number
  wonBets: number
  lostBets: number
  totalPayouts: number
  averageBetSize: number
  algorithmEfficiency: number
  liquidityUtilization: number
}

export function useAdminBetsOptimized() {
  const [bets, setBets] = useState<AdminBet[]>([])
  const [metrics, setMetrics] = useState<BetMetrics>({
    totalBets: 0,
    totalVolume: 0,
    seededVolume: 0,
    liveVolume: 0,
    matchedBets: 0,
    pendingBets: 0,
    wonBets: 0,
    lostBets: 0,
    totalPayouts: 0,
    averageBetSize: 0,
    algorithmEfficiency: 0,
    liquidityUtilization: 0
  })
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [page, setPage] = useState(1)
  const [hasMore, setHasMore] = useState(true)
  const [cache, setCache] = useState<Map<string, any>>(new Map())

  const PAGE_SIZE = 20
  const CACHE_DURATION = 3 * 60 * 1000 // 3 minutes

  const fetchBets = useCallback(async (pageNum = 1, status = 'all', forceRefresh = false) => {
    try {
      const cacheKey = `bets-${pageNum}-${status}`
      const cachedData = cache.get(cacheKey)
      
      // Return cached data if it exists and is not expired
      if (!forceRefresh && cachedData && Date.now() - cachedData.timestamp < CACHE_DURATION) {
        if (pageNum === 1) {
          setBets(cachedData.bets)
          setMetrics(cachedData.metrics)
        } else {
          setBets(prev => [...prev, ...cachedData.bets])
        }
        setLoading(false)
        return
      }

      setLoading(true)
      setError(null)

      // Simplified query with pagination
      let query = supabase
        .from('bets')
        .select(`
          id,
          user_id,
          market_id,
          outcome,
          amount,
          price,
          shares,
          status,
          seeded_amount,
          live_amount,
          potential_payout,
          created_at,
          updated_at,
          profiles(username, email),
          markets(title, status, yes_price, no_price, total_liquidity)
        `)
        .order('created_at', { ascending: false })
        .range((pageNum - 1) * PAGE_SIZE, pageNum * PAGE_SIZE - 1)

      if (status !== 'all') {
        query = query.eq('status', status)
      }

      const { data, error: fetchError } = await query

      if (fetchError) {
        throw fetchError
      }

      // Process bets with simplified algorithm metrics
      const processedBets = (data || []).map((bet) => {
        const market = Array.isArray(bet.markets) ? bet.markets[0] : bet.markets
        const userProfile = Array.isArray(bet.profiles) ? bet.profiles[0] : bet.profiles

        // Calculate simplified algorithm metrics
        const algorithmMetrics = calculateSimpleBetMetrics(bet, market)

        return {
          ...bet,
          user_profile: userProfile,
          market: market,
          algorithm_metrics: algorithmMetrics
        }
      })

      // Update cache
      const cacheData = {
        bets: processedBets,
        metrics: pageNum === 1 ? calculateMetrics(processedBets) : metrics,
        timestamp: Date.now()
      }
      setCache(prev => new Map(prev.set(cacheKey, cacheData)))

      if (pageNum === 1) {
        setBets(processedBets)
        setMetrics(cacheData.metrics)
      } else {
        setBets(prev => [...prev, ...processedBets])
      }

      setHasMore(processedBets.length === PAGE_SIZE)

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 
                          (err as any)?.message || 
                          (err as any)?.details || 
                          (err as any)?.hint || 
                          'Failed to fetch bets'
      setError(errorMessage)
      console.error('Error fetching bets:', {
        error: err,
        message: errorMessage,
        timestamp: new Date().toISOString()
      })
    } finally {
      setLoading(false)
    }
  }, [cache, metrics])

  const calculateSimpleBetMetrics = (bet: any, market: any) => {
    if (!market) {
      return {
        price_impact: 0,
        liquidity_utilization: 0,
        market_efficiency: 0,
        risk_score: 0
      }
    }

    // Simplified calculations
    const currentPrice = bet.outcome === 'yes' ? market.yes_price : market.no_price
    const priceImpact = Math.abs(currentPrice - 0.5) * 100

    const liquidityUtilization = market.total_liquidity > 0 
      ? ((bet.seeded_amount || 0) / market.total_liquidity) * 100 
      : 0

    const marketEfficiency = Math.max(0, 100 - priceImpact)

    const riskScore = market.total_liquidity > 0 
      ? Math.min(100, (bet.amount / market.total_liquidity) * 100)
      : 0

    return {
      price_impact: priceImpact,
      liquidity_utilization: liquidityUtilization,
      market_efficiency: marketEfficiency,
      risk_score: riskScore
    }
  }

  const calculateMetrics = (bets: AdminBet[]): BetMetrics => {
    const totalBets = bets.length
    const totalVolume = bets.reduce((sum, bet) => sum + bet.amount, 0)
    const seededVolume = bets.reduce((sum, bet) => sum + (bet.seeded_amount || 0), 0)
    const liveVolume = bets.reduce((sum, bet) => sum + (bet.live_amount || 0), 0)
    const matchedBets = bets.filter(bet => bet.status === 'matched').length
    const pendingBets = bets.filter(bet => bet.status === 'seeded' || bet.status === 'live').length
    const wonBets = bets.filter(bet => bet.status === 'won').length
    const lostBets = bets.filter(bet => bet.status === 'lost').length
    const totalPayouts = bets.reduce((sum, bet) => sum + (bet.potential_payout || 0), 0)
    const averageBetSize = totalBets > 0 ? totalVolume / totalBets : 0

    const algorithmEfficiency = bets.length > 0 
      ? bets.reduce((sum, bet) => sum + (bet.algorithm_metrics?.market_efficiency || 0), 0) / bets.length 
      : 0

    const totalLiquidity = bets.reduce((sum, bet) => sum + (bet.market?.total_liquidity || 0), 0)
    const liquidityUtilization = totalLiquidity > 0 ? (seededVolume / totalLiquidity) * 100 : 0

    return {
      totalBets,
      totalVolume,
      seededVolume,
      liveVolume,
      matchedBets,
      pendingBets,
      wonBets,
      lostBets,
      totalPayouts,
      averageBetSize,
      algorithmEfficiency,
      liquidityUtilization
    }
  }

  const loadMore = useCallback(() => {
    if (!loading && hasMore) {
      const nextPage = page + 1
      setPage(nextPage)
      fetchBets(nextPage)
    }
  }, [loading, hasMore, page, fetchBets])

  const refresh = useCallback(() => {
    setPage(1)
    setBets([])
    setCache(new Map())
    fetchBets(1, 'all', true)
  }, [fetchBets])

  const filterByStatus = useCallback((status: string) => {
    setPage(1)
    setBets([])
    fetchBets(1, status)
  }, [fetchBets])

  // Debounced search
  const [searchTerm, setSearchTerm] = useState('')
  const [debouncedSearchTerm, setDebouncedSearchTerm] = useState('')

  useEffect(() => {
    const timer = setTimeout(() => {
      setDebouncedSearchTerm(searchTerm)
    }, 300)

    return () => clearTimeout(timer)
  }, [searchTerm])

  const filteredBets = useMemo(() => {
    if (!debouncedSearchTerm) return bets
    
    return bets.filter(bet =>
      bet.user_profile?.username?.toLowerCase().includes(debouncedSearchTerm.toLowerCase()) ||
      bet.user_profile?.email?.toLowerCase().includes(debouncedSearchTerm.toLowerCase()) ||
      bet.market?.title?.toLowerCase().includes(debouncedSearchTerm.toLowerCase())
    )
  }, [bets, debouncedSearchTerm])

  useEffect(() => {
    fetchBets()
  }, [])

  return useMemo(() => ({
    bets: filteredBets,
    metrics,
    loading,
    error,
    hasMore,
    searchTerm,
    setSearchTerm,
    loadMore,
    refresh,
    filterByStatus
  }), [
    filteredBets,
    metrics,
    loading,
    error,
    hasMore,
    searchTerm,
    loadMore,
    refresh,
    filterByStatus
  ])
}
