'use client'

import { useState, useEffect, useCallback } from 'react'
import { supabase } from '@/lib/supabase-client'
import { Profile, WalletTransaction, Bet, Market, MarketWithStats } from '@/lib/types'

interface AdminStats {
  totalUsers: number
  totalMarkets: number
  totalBets: number
  totalVolume: number
  totalRevenue: number
  activeUsers: number
  activeMarkets: number
  pendingBets: number
  seededBets: number
  completedMarkets: number
}

interface UserWithStats extends Profile {
  totalBets: number
  totalVolume: number
  totalWinnings: number
  totalLosses: number
  netProfit: number
  winRate: number
  lastActive: string
}

// Note: This file uses a different MarketWithStats interface that extends Market
// For consistency, consider updating to use the shared MarketWithStats from @/lib/types

interface BetWithDetails extends Bet {
  user_email: string
  market_title: string
  market_status: string
  market_outcome: string | null
}

interface TransactionWithDetails extends WalletTransaction {
  user_email: string
  market_title?: string
  bet_outcome?: string
}

export function useAdminStats() {
  const [stats, setStats] = useState<AdminStats>({
    totalUsers: 0,
    totalMarkets: 0,
    totalBets: 0,
    totalVolume: 0,
    totalRevenue: 0,
    activeUsers: 0,
    activeMarkets: 0,
    pendingBets: 0,
    seededBets: 0,
    completedMarkets: 0
  })
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const fetchStats = useCallback(async () => {
    try {
      setLoading(true)
      setError(null)

      // Fetch all stats in parallel
      const [
        usersResult,
        marketsResult,
        betsResult,
        transactionsResult
      ] = await Promise.all([
        supabase.from('profiles').select('id, created_at, balance'),
        supabase.from('markets').select('id, status, total_volume, creator_earnings, admin_fee'),
        supabase.from('bets').select('id, status, amount, seeded_amount, created_at'),
        supabase.from('wallet_transactions').select('amount, type').eq('type', 'bet')
      ])

      if (usersResult.error) throw usersResult.error
      if (marketsResult.error) throw marketsResult.error
      if (betsResult.error) throw betsResult.error
      if (transactionsResult.error) throw transactionsResult.error

      const users = usersResult.data || []
      const markets = marketsResult.data || []
      const bets = betsResult.data || []
      const transactions = transactionsResult.error ? [] : (transactionsResult.data || [])

      // Calculate stats
      const totalVolume = markets.reduce((sum, market) => sum + (market.total_volume || 0), 0)
      const totalRevenue = markets.reduce((sum, market) => sum + (market.admin_fee || 0), 0)
      const activeUsers = users.filter(user => user.balance > 0).length
      const activeMarkets = markets.filter(market => market.status === 'active').length
      const pendingBets = bets.filter(bet => bet.status === 'live').length
      const seededBets = bets.filter(bet => bet.status === 'seeded').length
      const completedMarkets = markets.filter(market => market.status === 'resolved').length

      setStats({
        totalUsers: users.length,
        totalMarkets: markets.length,
        totalBets: bets.length,
        totalVolume,
        totalRevenue,
        activeUsers,
        activeMarkets,
        pendingBets,
        seededBets,
        completedMarkets
      })
    } catch (err) {
      console.error('Error fetching admin stats:', err)
      setError(err instanceof Error ? err.message : 'Failed to fetch stats')
    } finally {
      setLoading(false)
    }
  }, [])

  useEffect(() => {
    fetchStats()
  }, [fetchStats])

  return { stats, loading, error, refetch: fetchStats }
}

export function useAdminUsers() {
  const [users, setUsers] = useState<UserWithStats[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const fetchUsers = useCallback(async () => {
    try {
      setLoading(true)
      setError(null)

      // Fetch users with their betting stats
      const { data, error } = await supabase
        .from('profiles')
        .select(`
          *,
          bets!bets_user_id_fkey (
            id,
            amount,
            seeded_amount,
            status,
            created_at,
            markets (
              outcome,
              status
            )
          )
        `)
        .order('created_at', { ascending: false })

      if (error) throw error

      // Transform data to include stats
      const usersWithStats = (data || []).map(user => {
        const userBets = user.bets || []
        const totalBets = userBets.length
        const totalVolume = userBets.reduce((sum: number, bet: any) => sum + (bet.amount || 0) + (bet.seeded_amount || 0), 0)
        
        const wonBets = userBets.filter((bet: any) => 
          bet.markets?.outcome && bet.markets.outcome === bet.outcome
        )
        const lostBets = userBets.filter((bet: any) => 
          bet.markets?.outcome && bet.markets.outcome !== bet.outcome
        )
        
        const totalWinnings = wonBets.reduce((sum: number, bet: any) => sum + (bet.amount || 0) + (bet.seeded_amount || 0), 0)
        const totalLosses = lostBets.reduce((sum: number, bet: any) => sum + (bet.amount || 0) + (bet.seeded_amount || 0), 0)
        const netProfit = totalWinnings - totalLosses
        const winRate = totalBets > 0 ? (wonBets.length / totalBets) * 100 : 0
        
        const lastActive = userBets.length > 0 
          ? userBets.reduce((latest: string, bet: any) => 
              new Date(bet.created_at) > new Date(latest) ? bet.created_at : latest, 
              userBets[0].created_at
            )
          : user.created_at

        return {
          ...user,
          totalBets,
          totalVolume,
          totalWinnings,
          totalLosses,
          netProfit,
          winRate,
          lastActive
        }
      })

      setUsers(usersWithStats)
    } catch (err) {
      console.error('Error fetching users:', err)
      setError(err instanceof Error ? err.message : 'Failed to fetch users')
    } finally {
      setLoading(false)
    }
  }, [])

  useEffect(() => {
    fetchUsers()
  }, [fetchUsers])

  return { users, loading, error, refetch: fetchUsers }
}

export function useAdminMarkets() {
  const [markets, setMarkets] = useState<MarketWithStats[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const fetchMarkets = useCallback(async () => {
    try {
      setLoading(true)
      setError(null)

      const { data, error } = await supabase
        .from('markets')
        .select(`
          *,
          profiles!markets_creator_id_fkey (
            email
          ),
          bets (
            id,
            outcome,
            amount,
            seeded_amount,
            status
          )
        `)
        .order('created_at', { ascending: false })

      if (error) throw error

      // Transform data to include stats
      const marketsWithStats = (data || []).map(market => {
        const marketBets = market.bets || []
        const totalBets = marketBets.length
        const totalVolume = marketBets.reduce((sum: number, bet: any) => sum + (bet.amount || 0) + (bet.seeded_amount || 0), 0)
        const yesBets = marketBets.filter((bet: any) => bet.outcome === 'yes').length
        const noBets = marketBets.filter((bet: any) => bet.outcome === 'no').length
        const seededVolume = marketBets.reduce((sum: number, bet: any) => sum + (bet.seeded_amount || 0), 0)
        const liveVolume = marketBets.reduce((sum: number, bet: any) => sum + (bet.amount || 0), 0)

        return {
          ...market,
          creator_email: market.profiles?.email || 'Unknown',
          totalBets,
          totalVolume,
          yesBets,
          noBets,
          seededVolume,
          liveVolume,
          creatorEarnings: market.creator_earnings || 0,
          adminFees: market.admin_fee || 0
        }
      })

      setMarkets(marketsWithStats)
    } catch (err) {
      console.error('Error fetching markets:', err)
      setError(err instanceof Error ? err.message : 'Failed to fetch markets')
    } finally {
      setLoading(false)
    }
  }, [])

  useEffect(() => {
    fetchMarkets()
  }, [fetchMarkets])

  return { markets, loading, error, refetch: fetchMarkets }
}

export function useAdminBets() {
  const [bets, setBets] = useState<BetWithDetails[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const fetchBets = useCallback(async () => {
    try {
      setLoading(true)
      setError(null)

      const { data, error } = await supabase
        .from('bets')
        .select(`
          *,
          profiles!bets_user_id_fkey (
            email
          ),
          markets!bets_market_id_fkey (
            title,
            status,
            outcome
          )
        `)
        .order('created_at', { ascending: false })

      if (error) throw error

      // Transform data
      const betsWithDetails = (data || []).map(bet => ({
        ...bet,
        user_email: bet.profiles?.email || 'Unknown',
        market_title: bet.markets?.title || 'Unknown Market',
        market_status: bet.markets?.status || 'unknown',
        market_outcome: bet.markets?.outcome || null
      }))

      setBets(betsWithDetails)
    } catch (err) {
      console.error('Error fetching bets:', err)
      setError(err instanceof Error ? err.message : 'Failed to fetch bets')
    } finally {
      setLoading(false)
    }
  }, [])

  useEffect(() => {
    fetchBets()
  }, [fetchBets])

  return { bets, loading, error, refetch: fetchBets }
}

export function useAdminTransactions() {
  const [transactions, setTransactions] = useState<TransactionWithDetails[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const fetchTransactions = useCallback(async () => {
    try {
      setLoading(true)
      setError(null)

      // Fetch from unified transactions table
      const { data, error } = await supabase
        .from('unified_transactions')
        .select(`
          *,
          profiles!unified_transactions_user_id_fkey (
            email
          ),
          markets!unified_transactions_market_id_fkey (
            title
          )
        `)
        .order('created_at', { ascending: false })

      if (error) throw error

      // Transform data
      const transactionsWithDetails = (data || []).map(transaction => ({
        ...transaction,
        user_email: transaction.profiles?.email || 'Unknown',
        market_title: transaction.markets?.title || undefined,
        bet_outcome: transaction.description?.includes('YES') ? 'yes' : 
                     transaction.description?.includes('NO') ? 'no' : undefined,
        payment_method: transaction.payment_method
      }))

      setTransactions(transactionsWithDetails)
    } catch (err) {
      console.error('Error fetching transactions:', err)
      setError(err instanceof Error ? err.message : 'Failed to fetch transactions')
    } finally {
      setLoading(false)
    }
  }, [])

  useEffect(() => {
    fetchTransactions()
  }, [fetchTransactions])

  return { transactions, loading, error, refetch: fetchTransactions }
}
