'use client'

import { useState, useCallback, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'
import { MarketWithStats } from '@/lib/types'

interface MarketMetrics {
  totalMarkets: number
  activeMarkets: number
  resolvedMarkets: number
  totalVolume: number
  totalEarnings: number
  averageVolume: number
  resolutionRate: number
}

const PAGE_SIZE = 20
const CACHE_DURATION = 30000 // 30 seconds

export function useAdminMarketsOptimized() {
  const [markets, setMarkets] = useState<MarketWithStats[]>([])
  const [metrics, setMetrics] = useState<MarketMetrics>({
    totalMarkets: 0,
    activeMarkets: 0,
    resolvedMarkets: 0,
    totalVolume: 0,
    totalEarnings: 0,
    averageVolume: 0,
    resolutionRate: 0
  })
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [hasMore, setHasMore] = useState(true)
  const [searchTerm, setSearchTerm] = useState('')
  const [cache, setCache] = useState<{ data: MarketWithStats[], timestamp: number } | null>(null)

  const fetchMarkets = useCallback(async (refresh = false) => {
    console.log('useAdminMarketsOptimized: fetchMarkets called', { refresh })
    try {
      setLoading(true)
      setError(null)

      // Check cache first
      if (!refresh && cache && Date.now() - cache.timestamp < CACHE_DURATION) {
        setMarkets(cache.data)
        setLoading(false)
        return
      }

      // First, try a simple query to test if markets table exists
      console.log('Testing markets table access...')
      const { data: testData, error: testError } = await supabase
        .from('markets')
        .select('id, title, status')
        .limit(1)

      if (testError) {
        console.error('Markets table test failed:', {
          error: testError,
          message: testError.message,
          details: testError.details,
          hint: testError.hint,
          code: testError.code
        })
        throw testError
      }

      console.log('Markets table accessible, fetching full data...')

      // Fetch markets with basic stats (simplified query)
      console.log('Fetching markets data...')
      const { data, error } = await supabase
        .from('markets')
        .select(`
          id,
          title,
          description,
          status,
          admin_status,
          outcome,
          end_date,
          total_volume,
          yes_price,
          no_price,
          creator_id,
          creator_earnings,
          admin_fee,
          created_at,
          resolved_at,
          admin_notes,
          admin_action_at,
          admin_action_by,
          creator:profiles!markets_creator_id_fkey(username, email),
          admin_user:profiles!markets_admin_action_by_fkey(username)
        `)
        .order('created_at', { ascending: false })
        .limit(100) // Limit for performance

      if (error) {
        console.error('Supabase query error:', {
          error,
          message: error.message,
          details: error.details,
          hint: error.hint,
          code: error.code,
          timestamp: new Date().toISOString()
        })
        throw error
      }

      // Transform data
      const marketsWithStats: MarketWithStats[] = (data || []).map(market => ({
        id: market.id,
        title: market.title,
        description: market.description,
        status: market.status,
        admin_status: market.admin_status || 'pending',
        outcome: market.outcome,
        end_date: market.end_date,
        total_volume: market.total_volume || 0,
        yes_price: market.yes_price || 0,
        no_price: market.no_price || 0,
        creator_id: market.creator_id,
        creator_earnings: market.creator_earnings || 0,
        admin_fee: market.admin_fee || 0,
        created_at: market.created_at,
        resolved_at: market.resolved_at,
        admin_notes: market.admin_notes,
        admin_action_at: market.admin_action_at,
        creator_email: market.creator?.email || 'Unknown',
        admin_username: market.admin_user?.username || null,
        admin_status_display: market.admin_status || 'Pending',
        totalBets: 0, // Will be calculated separately if needed
        yesBets: 0,
        noBets: 0,
        seededVolume: 0,
        liveVolume: 0
      }))

      // Calculate metrics
      const totalMarkets = marketsWithStats.length
      const activeMarkets = marketsWithStats.filter(m => m.status === 'active').length
      const resolvedMarkets = marketsWithStats.filter(m => m.status === 'resolved').length
      const totalVolume = marketsWithStats.reduce((sum, m) => sum + m.total_volume, 0)
      const totalEarnings = marketsWithStats.reduce((sum, m) => sum + m.creator_earnings + m.admin_fee, 0)
      const averageVolume = totalMarkets > 0 ? totalVolume / totalMarkets : 0
      const resolutionRate = totalMarkets > 0 ? (resolvedMarkets / totalMarkets) * 100 : 0

      setMetrics({
        totalMarkets,
        activeMarkets,
        resolvedMarkets,
        totalVolume,
        totalEarnings,
        averageVolume,
        resolutionRate
      })

      setMarkets(marketsWithStats)
      setCache({ data: marketsWithStats, timestamp: Date.now() })
      setHasMore(marketsWithStats.length >= 100)

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 
                          (err as any)?.message || 
                          (err as any)?.details || 
                          (err as any)?.hint || 
                          'Failed to fetch markets'
      setError(errorMessage)
      console.error('Error fetching markets:', {
        error: err,
        message: errorMessage,
        timestamp: new Date().toISOString()
      })
    } finally {
      setLoading(false)
    }
  }, [cache])

  const loadMore = useCallback(async () => {
    if (!hasMore || loading) return

    try {
      const { data, error } = await supabase
        .from('markets')
        .select(`
          id,
          title,
          description,
          status,
          admin_status,
          outcome,
          end_date,
          total_volume,
          yes_price,
          no_price,
          creator_id,
          creator_earnings,
          admin_fee,
          created_at,
          resolved_at,
          admin_notes,
          admin_action_at,
          admin_action_by,
          creator:profiles!markets_creator_id_fkey(username, email),
          admin_user:profiles!markets_admin_action_by_fkey(username)
        `)
        .order('created_at', { ascending: false })
        .range(markets.length, markets.length + PAGE_SIZE - 1)

      if (error) throw error

      const newMarkets: MarketWithStats[] = (data || []).map(market => ({
        id: market.id,
        title: market.title,
        description: market.description,
        status: market.status,
        admin_status: market.admin_status || 'pending',
        outcome: market.outcome,
        end_date: market.end_date,
        total_volume: market.total_volume || 0,
        yes_price: market.yes_price || 0,
        no_price: market.no_price || 0,
        creator_id: market.creator_id,
        creator_earnings: market.creator_earnings || 0,
        admin_fee: market.admin_fee || 0,
        created_at: market.created_at,
        resolved_at: market.resolved_at,
        admin_notes: market.admin_notes,
        admin_action_at: market.admin_action_at,
        creator_email: market.creator?.email || 'Unknown',
        admin_username: market.admin_user?.username || null,
        admin_status_display: market.admin_status || 'Pending',
        totalBets: 0,
        yesBets: 0,
        noBets: 0,
        seededVolume: 0,
        liveVolume: 0
      }))

      setMarkets(prev => [...prev, ...newMarkets])
      setHasMore(newMarkets.length === PAGE_SIZE)

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 
                          (err as any)?.message || 
                          (err as any)?.details || 
                          (err as any)?.hint || 
                          'Failed to load more markets'
      console.error('Error loading more markets:', {
        error: err,
        message: errorMessage,
        timestamp: new Date().toISOString()
      })
    }
  }, [markets.length, hasMore, loading])

  const filterByStatus = useCallback((status: string) => {
    // This would filter the current markets array
    // For now, we'll just trigger a refresh with the filter
    fetchMarkets(true)
  }, [fetchMarkets])

  const refresh = useCallback(() => {
    fetchMarkets(true)
  }, [fetchMarkets])

  useEffect(() => {
    console.log('useAdminMarketsOptimized: useEffect calling fetchMarkets')
    fetchMarkets(false)
  }, [fetchMarkets])

  // Filter markets based on search term
  const filteredMarkets = markets.filter(market =>
    market.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
    market.description.toLowerCase().includes(searchTerm.toLowerCase()) ||
    market.creator_email.toLowerCase().includes(searchTerm.toLowerCase()) ||
    market.status.toLowerCase().includes(searchTerm.toLowerCase())
  )

  return {
    markets: filteredMarkets,
    metrics,
    loading,
    error,
    hasMore,
    searchTerm,
    setSearchTerm,
    loadMore,
    refresh,
    filterByStatus
  }
}