'use client'

import { useState, useCallback, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'

interface TransactionWithDetails {
  id: string
  user_id: string
  type: string
  amount: number
  currency: string
  status: string
  description: string
  created_at: string
  processed_at: string | null
  external_transaction_id: string | null
  payment_method: any
  market_id: string | null
  user_email: string
  market_title?: string
  fee_amount: number
  net_amount: number
}

interface TransactionMetrics {
  totalTransactions: number
  totalVolume: number
  deposits: number
  withdrawals: number
  bets: number
  winnings: number
  fees: number
  averageAmount: number
  completionRate: number
}

const PAGE_SIZE = 25
const CACHE_DURATION = 30000 // 30 seconds

export function useAdminTransactionsOptimized() {
  const [transactions, setTransactions] = useState<TransactionWithDetails[]>([])
  const [metrics, setMetrics] = useState<TransactionMetrics>({
    totalTransactions: 0,
    totalVolume: 0,
    deposits: 0,
    withdrawals: 0,
    bets: 0,
    winnings: 0,
    fees: 0,
    averageAmount: 0,
    completionRate: 0
  })
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [hasMore, setHasMore] = useState(true)
  const [searchTerm, setSearchTerm] = useState('')
  const [cache, setCache] = useState<{ data: TransactionWithDetails[], timestamp: number } | null>(null)

  const fetchTransactions = useCallback(async (refresh = false) => {
    try {
      setLoading(true)
      setError(null)

      // Check cache first
      if (!refresh && cache && Date.now() - cache.timestamp < CACHE_DURATION) {
        setTransactions(cache.data)
        setLoading(false)
        return
      }

      // Fetch transactions with user details
      const { data, error } = await supabase
        .from('unified_transactions')
        .select(`
          id,
          user_id,
          type,
          amount,
          currency,
          status,
          description,
          created_at,
          processed_at,
          external_transaction_id,
          payment_method,
          market_id,
          fee_amount,
          net_amount,
          profiles!unified_transactions_user_id_fkey(email),
          markets!unified_transactions_market_id_fkey(title)
        `)
        .order('created_at', { ascending: false })
        .limit(100) // Limit for performance

      if (error) throw error

      // Transform data
      const transactionsWithDetails: TransactionWithDetails[] = (data || []).map(transaction => ({
        id: transaction.id,
        user_id: transaction.user_id,
        type: transaction.type,
        amount: transaction.amount || 0,
        currency: transaction.currency || 'USD',
        status: transaction.status || 'pending',
        description: transaction.description || '',
        created_at: transaction.created_at,
        processed_at: transaction.processed_at,
        external_transaction_id: transaction.external_transaction_id,
        payment_method: transaction.payment_method,
        market_id: transaction.market_id,
        fee_amount: transaction.fee_amount || 0,
        net_amount: transaction.net_amount || transaction.amount || 0,
        user_email: (transaction.profiles as any)?.email || 'Unknown',
        market_title: (transaction.markets as any)?.title
      }))

      // Calculate metrics
      const totalTransactions = transactionsWithDetails.length
      const totalVolume = transactionsWithDetails.reduce((sum, t) => sum + Math.abs(t.amount), 0)
      const deposits = transactionsWithDetails.filter(t => t.type === 'deposit').length
      const withdrawals = transactionsWithDetails.filter(t => t.type === 'withdrawal').length
      const bets = transactionsWithDetails.filter(t => t.type === 'bet').length
      const winnings = transactionsWithDetails.filter(t => t.type === 'win').length
      const fees = transactionsWithDetails.reduce((sum, t) => sum + (t.fee_amount || 0), 0)
      const averageAmount = totalTransactions > 0 ? totalVolume / totalTransactions : 0
      const completedTransactions = transactionsWithDetails.filter(t => t.status === 'completed').length
      const completionRate = totalTransactions > 0 ? (completedTransactions / totalTransactions) * 100 : 0

      setMetrics({
        totalTransactions,
        totalVolume,
        deposits,
        withdrawals,
        bets,
        winnings,
        fees,
        averageAmount,
        completionRate
      })

      setTransactions(transactionsWithDetails)
      setCache({ data: transactionsWithDetails, timestamp: Date.now() })
      setHasMore(transactionsWithDetails.length >= 100)

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 
                          (err as any)?.message || 
                          (err as any)?.details || 
                          (err as any)?.hint || 
                          'Failed to fetch transactions'
      setError(errorMessage)
      console.error('Error fetching transactions:', {
        error: err,
        message: errorMessage,
        timestamp: new Date().toISOString()
      })
    } finally {
      setLoading(false)
    }
  }, [cache])

  const loadMore = useCallback(async () => {
    if (!hasMore || loading) return

    try {
      const { data, error } = await supabase
        .from('unified_transactions')
        .select(`
          id,
          user_id,
          type,
          amount,
          currency,
          status,
          description,
          created_at,
          processed_at,
          external_transaction_id,
          payment_method,
          market_id,
          fee_amount,
          net_amount,
          profiles!unified_transactions_user_id_fkey(email),
          markets!unified_transactions_market_id_fkey(title)
        `)
        .order('created_at', { ascending: false })
        .range(transactions.length, transactions.length + PAGE_SIZE - 1)

      if (error) throw error

      const newTransactions: TransactionWithDetails[] = (data || []).map(transaction => ({
        id: transaction.id,
        user_id: transaction.user_id,
        type: transaction.type,
        amount: transaction.amount || 0,
        currency: transaction.currency || 'USD',
        status: transaction.status || 'pending',
        description: transaction.description || '',
        created_at: transaction.created_at,
        processed_at: transaction.processed_at,
        external_transaction_id: transaction.external_transaction_id,
        payment_method: transaction.payment_method,
        market_id: transaction.market_id,
        fee_amount: transaction.fee_amount || 0,
        net_amount: transaction.net_amount || transaction.amount || 0,
        user_email: (transaction.profiles as any)?.email || 'Unknown',
        market_title: (transaction.markets as any)?.title
      }))

      setTransactions(prev => [...prev, ...newTransactions])
      setHasMore(newTransactions.length === PAGE_SIZE)

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 
                          (err as any)?.message || 
                          (err as any)?.details || 
                          (err as any)?.hint || 
                          'Failed to load more transactions'
      console.error('Error loading more transactions:', {
        error: err,
        message: errorMessage,
        timestamp: new Date().toISOString()
      })
    }
  }, [transactions.length, hasMore, loading])

  const filterByType = useCallback((type: string) => {
    // This would filter the current transactions array
    // For now, we'll just trigger a refresh
    fetchTransactions(true)
  }, [fetchTransactions])

  const refresh = useCallback(() => {
    fetchTransactions(true)
  }, [fetchTransactions])

  useEffect(() => {
    fetchTransactions()
  }, [fetchTransactions])

  // Filter transactions based on search term
  const filteredTransactions = transactions.filter(transaction =>
    transaction.user_email.toLowerCase().includes(searchTerm.toLowerCase()) ||
    transaction.type.toLowerCase().includes(searchTerm.toLowerCase()) ||
    transaction.description.toLowerCase().includes(searchTerm.toLowerCase()) ||
    transaction.id.toLowerCase().includes(searchTerm.toLowerCase()) ||
    (transaction.market_title && transaction.market_title.toLowerCase().includes(searchTerm.toLowerCase()))
  )

  return {
    transactions: filteredTransactions,
    metrics,
    loading,
    error,
    hasMore,
    searchTerm,
    setSearchTerm,
    loadMore,
    refresh,
    filterByType
  }
}
