import { useState, useEffect, useCallback, useMemo } from 'react'
import { supabase } from '@/lib/supabase-client'

export interface AdminUser {
  id: string
  username: string
  email: string
  balance: number
  is_admin: boolean
  is_banned: boolean
  kyc_status: string
  kyc_verified: boolean
  created_at: string
  last_login?: string
  bet_count: number
  win_rate: number
  total_profit: number
  risk_score: number
  total_volume: number
  last_activity: string
}

export interface UserMetrics {
  totalUsers: number
  activeUsers: number
  bannedUsers: number
  kycPending: number
  kycApproved: number
  totalVolume: number
  avgWinRate: number
  topPerformers: number
}

export function useAdminUsersOptimized() {
  const [users, setUsers] = useState<AdminUser[]>([])
  const [metrics, setMetrics] = useState<UserMetrics>({
    totalUsers: 0,
    activeUsers: 0,
    bannedUsers: 0,
    kycPending: 0,
    kycApproved: 0,
    totalVolume: 0,
    avgWinRate: 0,
    topPerformers: 0
  })
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)
  const [page, setPage] = useState(1)
  const [hasMore, setHasMore] = useState(true)
  const [cache, setCache] = useState<Map<string, any>>(new Map())

  const PAGE_SIZE = 20
  const CACHE_DURATION = 5 * 60 * 1000 // 5 minutes

  const fetchUsers = useCallback(async (pageNum = 1, status = 'all', forceRefresh = false) => {
    console.log('useAdminUsersOptimized: fetchUsers called', { pageNum, status, forceRefresh })
    try {
      const cacheKey = `users-${status}`
      const cachedData = cache.get(cacheKey)
      
      // Return cached data if it exists and is not expired
      if (!forceRefresh && cachedData && Date.now() - cachedData.timestamp < CACHE_DURATION) {
        setUsers(cachedData.users)
        setMetrics(cachedData.metrics)
        setLoading(false)
        return
      }

      setLoading(true)
      setError(null)

      // Fetch ALL users from the database
      console.log('Fetching ALL users from profiles table...')
      let query = supabase
        .from('profiles')
        .select(`
          id,
          username,
          email,
          balance,
          is_admin,
          is_banned,
          kyc_status,
          kyc_verified,
          created_at,
          last_login
        `)
        .order('created_at', { ascending: false })

      if (status !== 'all') {
        if (status === 'banned') {
          query = query.eq('is_banned', true)
        } else if (status === 'admin') {
          query = query.eq('is_admin', true)
        } else if (status === 'kyc_pending') {
          query = query.eq('kyc_status', 'pending')
        } else if (status === 'kyc_approved') {
          query = query.eq('kyc_status', 'approved')
        }
      }

      const { data, error: fetchError } = await query

      if (fetchError) {
        throw fetchError
      }

      // Get user stats in parallel (simplified)
      const usersWithStats = await Promise.all(
        (data || []).map(async (user) => {
          try {
            // Get basic stats with a single query
            const { data: stats } = await supabase
              .from('bets')
              .select('amount, status, potential_payout, created_at')
              .eq('user_id', user.id)
              .limit(50) // Limit to recent bets for performance

            const betCount = stats?.length || 0
            const wonBets = stats?.filter(bet => bet.status === 'won').length || 0
            const totalCompletedBets = stats?.filter(bet => bet.status === 'won' || bet.status === 'lost').length || 0
            const winRate = totalCompletedBets > 0 ? (wonBets / totalCompletedBets) * 100 : 0
            
            const totalProfit = stats?.reduce((sum, bet) => {
              if (bet.status === 'won') {
                return sum + (bet.potential_payout || 0)
              } else if (bet.status === 'lost') {
                return sum - (bet.amount || 0)
              }
              return sum
            }, 0) || 0

            const totalVolume = stats?.reduce((sum, bet) => sum + (bet.amount || 0), 0) || 0

            // Simple risk score calculation
            const riskScore = calculateSimpleRiskScore(stats || [], totalVolume, betCount)

            // Get last activity
            const lastBet = stats?.sort((a, b) => new Date(b.created_at).getTime() - new Date(a.created_at).getTime())[0]
            const lastActivity = lastBet ? lastBet.created_at : user.created_at

            return {
              ...user,
              bet_count: betCount,
              win_rate: winRate,
              total_profit: totalProfit,
              total_volume: totalVolume,
              risk_score: riskScore,
              last_activity: lastActivity
            }
          } catch (err) {
            console.error(`Error getting stats for user ${user.id}:`, {
              error: err,
              userId: user.id,
              timestamp: new Date().toISOString()
            })
            return {
              ...user,
              bet_count: 0,
              win_rate: 0,
              total_profit: 0,
              total_volume: 0,
              risk_score: 0,
              last_activity: user.created_at
            }
          }
        })
      )

      // Update cache
      const cacheData = {
        users: usersWithStats,
        metrics: calculateMetrics(usersWithStats),
        timestamp: Date.now()
      }
      setCache(prev => new Map(prev.set(cacheKey, cacheData)))

      // Always set all users (no pagination)
      setUsers(usersWithStats)
      setMetrics(cacheData.metrics)
      setHasMore(false) // No more pages since we fetch all users

    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : 
                          (err as any)?.message || 
                          (err as any)?.details || 
                          (err as any)?.hint || 
                          'Failed to fetch users'
      setError(errorMessage)
      console.error('Error fetching users:', {
        error: err,
        message: errorMessage,
        timestamp: new Date().toISOString()
      })
    } finally {
      setLoading(false)
    }
  }, [cache, metrics])

  const calculateSimpleRiskScore = (bets: any[], totalVolume: number, betCount: number): number => {
    if (betCount === 0) return 0

    // Simplified risk calculation
    const avgBet = totalVolume / betCount
    const largeBets = bets.filter(bet => (bet.amount || 0) > avgBet * 2).length
    const sizeRisk = largeBets / betCount

    // Calculate frequency risk
    const now = new Date()
    const recentBets = bets.filter(bet => 
      new Date(bet.created_at) > new Date(now.getTime() - 24 * 60 * 60 * 1000)
    ).length
    const frequencyRisk = Math.min(recentBets / 10, 1)

    return Math.round((sizeRisk * 50 + frequencyRisk * 50))
  }

  const calculateMetrics = (users: AdminUser[]): UserMetrics => {
    const totalUsers = users.length
    const activeUsers = users.filter(u => !u.is_banned && new Date(u.last_activity) > new Date(Date.now() - 30 * 24 * 60 * 60 * 1000)).length
    const bannedUsers = users.filter(u => u.is_banned).length
    const kycPending = users.filter(u => u.kyc_status === 'pending').length
    const kycApproved = users.filter(u => u.kyc_status === 'approved').length
    const totalVolume = users.reduce((sum, u) => sum + u.total_volume, 0)
    const avgWinRate = totalUsers > 0 ? users.reduce((sum, u) => sum + u.win_rate, 0) / totalUsers : 0
    const topPerformers = users.filter(u => u.win_rate > 70 && u.bet_count > 10).length

    return {
      totalUsers,
      activeUsers,
      bannedUsers,
      kycPending,
      kycApproved,
      totalVolume,
      avgWinRate,
      topPerformers
    }
  }

  const loadMore = useCallback(() => {
    // No-op since we fetch all users at once
    console.log('loadMore called but all users are already loaded')
  }, [])

  const refresh = useCallback(() => {
    setUsers([])
    setCache(new Map())
    fetchUsers(1, 'all', true)
  }, [fetchUsers])

  const filterByStatus = useCallback((status: string) => {
    setUsers([])
    fetchUsers(1, status)
  }, [fetchUsers])

  // Debounced search
  const [searchTerm, setSearchTerm] = useState('')
  const [debouncedSearchTerm, setDebouncedSearchTerm] = useState('')

  useEffect(() => {
    const timer = setTimeout(() => {
      setDebouncedSearchTerm(searchTerm)
    }, 300)

    return () => clearTimeout(timer)
  }, [searchTerm])

  const filteredUsers = useMemo(() => {
    if (!debouncedSearchTerm) return users
    
    return users.filter(user =>
      user.username?.toLowerCase().includes(debouncedSearchTerm.toLowerCase()) ||
      user.email?.toLowerCase().includes(debouncedSearchTerm.toLowerCase())
    )
  }, [users, debouncedSearchTerm])

  useEffect(() => {
    console.log('useAdminUsersOptimized: useEffect calling fetchUsers')
    fetchUsers(1, 'all', false)
  }, [])

  return useMemo(() => ({
    users: filteredUsers,
    metrics,
    loading,
    error,
    hasMore,
    searchTerm,
    setSearchTerm,
    loadMore,
    refresh,
    filterByStatus
  }), [
    filteredUsers,
    metrics,
    loading,
    error,
    hasMore,
    searchTerm,
    loadMore,
    refresh,
    filterByStatus
  ])
}
