'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'
import { Market } from '@/lib/types'

export function useHomeMarkets(limit: number = 20) {
  const [markets, setMarkets] = useState<Market[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    const fetchLiveAndScheduledMarkets = async (isBackground = false) => {
      try {
        // Only show loading on initial fetch, not background updates
        if (!isBackground) {
          setLoading(true)
          setError(null)
        }

        // Fetch only live and scheduled markets (exclude closed, resolved, and ended markets)
        const now = new Date().toISOString()
        const { data: marketsData, error: marketsError } = await supabase
          .from('markets')
          .select('*')
          .in('status', ['active', 'pending']) // Only live (active) and scheduled (pending) markets
          .neq('status', 'closed') // Explicitly exclude closed markets
          .neq('status', 'resolved') // Explicitly exclude resolved markets
          .gt('end_date', now) // Only show markets that haven't ended yet
          .order('created_at', { ascending: false })
          .limit(limit)

        if (marketsError) throw marketsError

        // Calculate pool information for each market
        const marketsWithPools = await Promise.all(
          (marketsData || []).map(async (market) => {
            if (market.market_type === 'multi_outcome') {
              // Handle multi-outcome markets
              const { data: outcomes } = await supabase
                .from('market_outcomes')
                .select('*')
                .eq('market_id', market.id)
                .order('sort_order', { ascending: true })

              const { data: bets } = await supabase
                .from('bets')
                .select('amount, outcome, outcome_id')
                .eq('market_id', market.id)
                .eq('status', 'matched')

              // Calculate total volume from bets
              const totalVolume = bets?.reduce((sum: number, bet: any) => sum + (bet.amount || 0), 0) || 0

              // Update outcomes with calculated prices
              const updatedOutcomes = outcomes?.map(outcome => {
                const outcomeBets = bets?.filter((bet: any) => bet.outcome_id === outcome.id) || []
                const outcomeVolume = outcomeBets.reduce((sum: number, bet: any) => sum + (bet.amount || 0), 0)
                
                return {
                  ...outcome,
                  volume: outcomeVolume,
                  price: market.total_liquidity > 0 ? Math.max(0, Math.min(1, outcome.pool / market.total_liquidity)) : 0
                }
              }) || []

              return {
                ...market,
                outcomes: updatedOutcomes,
                total_volume: totalVolume,
                total_liquidity: market.total_liquidity || 0
              }
            } else {
              // Handle binary markets
              const { data: bets } = await supabase
                .from('bets')
                .select('seeded_amount, live_amount, outcome')
                .eq('market_id', market.id)
                .eq('status', 'matched')

              // Calculate pools for each outcome and type
              let yesSeededPool = 0
              let noSeededPool = 0
              let yesLivePool = 0
              let noLivePool = 0
              let totalSeededVolume = 0
              let totalLiveVolume = 0

              bets?.forEach(bet => {
                const seededAmount = bet.seeded_amount || 0
                const liveAmount = bet.live_amount || 0
                
                totalSeededVolume += seededAmount
                totalLiveVolume += liveAmount
                
                if (bet.outcome === 'yes') {
                  yesSeededPool += seededAmount
                  yesLivePool += liveAmount
                } else {
                  noSeededPool += seededAmount
                  noLivePool += liveAmount
                }
              })

              // Use the market's stored prices (calculated by database functions)
              // These should already be correctly calculated based on actual trading data
              const yesPrice = Math.max(0, Math.min(1, market.yes_price || 0.5))
              const noPrice = Math.max(0, Math.min(1, market.no_price || 0.5))
              
              // Calculate pools for display purposes
              const totalYesPool = yesSeededPool + yesLivePool
              const totalNoPool = noSeededPool + noLivePool

              return {
                ...market,
                yes_price: yesPrice,
                no_price: noPrice,
                yes_pool: totalYesPool,
                no_pool: totalNoPool,
                seeded_volume: totalSeededVolume,
                live_volume: totalLiveVolume,
                total_volume: totalSeededVolume + totalLiveVolume,
                total_liquidity: market.total_liquidity || (market.yes_pool || 0) + (market.no_pool || 0)
              }
            }
          })
        )

        setMarkets(marketsWithPools)
      } catch (err) {
        // Only set error on initial fetch, not background updates
        if (!isBackground) {
          setError(err instanceof Error ? err.message : 'Failed to fetch markets')
        }
        console.error('Error fetching live and scheduled markets:', err)
      } finally {
        // Only set loading to false on initial fetch
        if (!isBackground) {
          setLoading(false)
        }
      }
    }

    // Initial fetch (with loading state)
    fetchLiveAndScheduledMarkets(false)

    // Set up background updates every 5 seconds (without loading state)
    const interval = setInterval(() => {
      fetchLiveAndScheduledMarkets(true)
    }, 5000)

    // Cleanup interval on unmount
    return () => clearInterval(interval)
  }, [limit])

  return { markets, loading, error }
}

// Additional hook for real-time market updates
export function useLiveMarketUpdates(marketIds: string[]) {
  const [updates, setUpdates] = useState<Record<string, any>>({})

  useEffect(() => {
    if (marketIds.length === 0) return

    // Set up real-time subscription for market updates
    const subscription = supabase
      .channel('market-updates')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'markets',
          filter: `id=in.(${marketIds.join(',')})`
        },
        (payload) => {
          console.log('Market update received:', payload)
          setUpdates(prev => ({
            ...prev,
            [(payload.new as any)?.id || '']: {
              ...payload.new,
              updated_at: new Date().toISOString()
            }
          }))
        }
      )
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'bets',
          filter: `market_id=in.(${marketIds.join(',')})`
        },
        (payload) => {
          console.log('Bet update received:', payload)
          // Trigger a refetch of market data when bets change
          setUpdates(prev => ({
            ...prev,
            [(payload.new as any)?.market_id || '']: {
              ...prev[(payload.new as any)?.market_id || ''],
              bet_updated: true,
              updated_at: new Date().toISOString()
            }
          }))
        }
      )
      .subscribe()

    return () => {
      subscription.unsubscribe()
    }
  }, [marketIds])

  return updates
}

// Hook for market price history updates
export function useMarketPriceHistory(marketId: string) {
  const [priceHistory, setPriceHistory] = useState<any[]>([])

  useEffect(() => {
    if (!marketId) return

    const fetchPriceHistory = async () => {
      try {
        const { data, error } = await supabase
          .from('price_history')
          .select('*')
          .eq('market_id', marketId)
          .order('created_at', { ascending: true })
          .limit(100) // Last 100 price points

        if (!error && data) {
          setPriceHistory(data)
        }
      } catch (err) {
        console.error('Error fetching price history:', err)
      }
    }

    fetchPriceHistory()

    // Set up real-time subscription for price history updates
    const subscription = supabase
      .channel(`price-history-${marketId}`)
      .on(
        'postgres_changes',
        {
          event: 'INSERT',
          schema: 'public',
          table: 'price_history',
          filter: `market_id=eq.${marketId}`
        },
        (payload) => {
          console.log('Price history update received:', payload)
          setPriceHistory(prev => [...prev, payload.new].slice(-100)) // Keep last 100 points
        }
      )
      .subscribe()

    return () => {
      subscription.unsubscribe()
    }
  }, [marketId])

  return priceHistory
}
