'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'

interface LeaderboardEntry {
  rank: number
  username: string
  points: number
  win_rate: number
  total_bets: number
  profit: number
}

export function useLeaderboard(timePeriod: 'all' | 'month' | 'week' = 'month') {
  const [leaderboard, setLeaderboard] = useState<LeaderboardEntry[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    const fetchLeaderboard = async () => {
      try {
        setLoading(true)
        setError(null)

        // Calculate date range based on time period
        let dateFilter = ''
        const now = new Date()
        
        if (timePeriod === 'week') {
          const weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000)
          dateFilter = `AND b.created_at >= '${weekAgo.toISOString()}'`
        } else if (timePeriod === 'month') {
          const monthAgo = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000)
          dateFilter = `AND b.created_at >= '${monthAgo.toISOString()}'`
        }
        // 'all' time period uses no date filter

        // Try to fetch from leaderboard view first
        const { data: leaderboardData, error: leaderboardError } = await supabase
          .from('leaderboard')
          .select('*')
          .order('total_points', { ascending: false })
          .limit(10)

        if (leaderboardError) {
          console.warn('Leaderboard view not available, calculating manually:', leaderboardError)
          
          // Fallback: calculate leaderboard manually from profiles and bets
          const { data: profiles, error: profilesError } = await supabase
            .from('profiles')
            .select('id, username, total_points')
            .eq('account_status', 'active')
            .order('total_points', { ascending: false })
            .limit(10)

          if (profilesError) {
            console.warn('Failed to fetch profiles, using mock data:', profilesError)
            throw new Error('Database connection failed')
          }

          // Calculate real stats for each user from their bets
          const calculatedLeaderboard = await Promise.all(
            (profiles || []).map(async (profile, index) => {
              // Fetch user's bets with date filter
              let query = supabase
                .from('bets')
                .select('status, amount, potential_payout, created_at')
                .eq('user_id', profile.id)
              
              // Apply date filter if not 'all' time period
              if (timePeriod !== 'all') {
                const now = new Date()
                if (timePeriod === 'week') {
                  const weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000)
                  query = query.gte('created_at', weekAgo.toISOString())
                } else if (timePeriod === 'month') {
                  const monthAgo = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000)
                  query = query.gte('created_at', monthAgo.toISOString())
                }
              }
              
              const { data: userBets, error: betsError } = await query

              if (betsError) {
                console.warn(`Failed to fetch bets for user ${profile.id}:`, betsError)
              }

              const bets = userBets || []
              
              // Calculate real win rate
              const wonBets = bets.filter(bet => bet.status === 'won').length
              const totalCompletedBets = bets.filter(bet => bet.status === 'won' || bet.status === 'lost').length
              const winRate = totalCompletedBets > 0 ? Math.round((wonBets / totalCompletedBets) * 100) : 0

              // Calculate real total bets
              const totalBets = bets.length

              // Calculate real profit from won bets
              const profit = bets.reduce((sum, bet) => {
                if (bet.status === 'won') {
                  return sum + (bet.potential_payout || 0)
                } else if (bet.status === 'lost') {
                  return sum - (bet.amount || 0)
                }
                return sum
              }, 0)

              return {
                rank: index + 1,
                username: profile.username || `User${profile.id.slice(0, 8)}`,
                points: profile.total_points || 0,
                win_rate: winRate,
                total_bets: totalBets,
                profit: profit
              }
            })
          )

          setLeaderboard(calculatedLeaderboard)
        } else {
          // Use leaderboard view data
          const formattedLeaderboard = leaderboardData?.map((entry, index) => ({
            rank: index + 1,
            username: entry.username || `User${entry.user_id?.slice(0, 8)}`,
            points: entry.total_points || 0,
            win_rate: Math.round(entry.win_rate || 0),
            total_bets: entry.total_bets || 0,
            profit: Math.round(entry.total_profit || 0)
          })) || []

          setLeaderboard(formattedLeaderboard)
        }
      } catch (err) {
        console.error('Error fetching leaderboard:', err)
        
        // Fallback to mock data if all else fails (database unavailable)
        console.warn('Using mock data as fallback - database connection failed')
        const mockData = [
          { rank: 1, username: 'Database unavailable', points: 0, win_rate: 0, total_bets: 0, profit: 0 },
          { rank: 2, username: 'Please check connection', points: 0, win_rate: 0, total_bets: 0, profit: 0 },
          { rank: 3, username: 'Real data will appear', points: 0, win_rate: 0, total_bets: 0, profit: 0 },
        ]
        setLeaderboard(mockData)
        setError('Unable to load leaderboard data. Please check your connection.')
      } finally {
        setLoading(false)
      }
    }

    fetchLeaderboard()
  }, [timePeriod])

  return { leaderboard, loading, error }
}