'use client'

import { useState, useEffect } from 'react'
import { supabase } from '@/lib/supabase-client'

interface LiveBalanceData {
  balance: number
  lastUpdated: Date
  isConnected: boolean
}

export function useLiveBalance(userId?: string) {
  const [data, setData] = useState<LiveBalanceData>({
    balance: 0,
    lastUpdated: new Date(),
    isConnected: false
  })

  useEffect(() => {
    if (!userId) return

    // Initial fetch
    const fetchInitialBalance = async () => {
      try {
        const { data: profile, error } = await supabase
          .from('profiles')
          .select('balance')
          .eq('id', userId)
          .single()

        if (!error && profile) {
          setData(prev => ({
            ...prev,
            balance: profile.balance || 0,
            lastUpdated: new Date()
          }))
        }
      } catch (err) {
        console.error('Error fetching initial balance:', err)
      }
    }

    fetchInitialBalance()

    // Set up real-time subscription
    const subscription = supabase
      .channel('live-balance-updates')
      .on(
        'postgres_changes',
        {
          event: 'UPDATE',
          schema: 'public',
          table: 'profiles',
          filter: `id=eq.${userId}`
        },
        (payload) => {
          console.log('Balance updated:', payload)
          setData(prev => ({
            balance: payload.new.balance || 0,
            lastUpdated: new Date(),
            isConnected: true
          }))
        }
      )
      .on('system', {}, (status) => {
        console.log('Supabase connection status:', status)
        setData(prev => ({
          ...prev,
          isConnected: status === 'SUBSCRIBED'
        }))
      })
      .subscribe()

    // Cleanup subscription
    return () => {
      subscription.unsubscribe()
    }
  }, [userId])

  return data
}
