'use client'

import { useState, useEffect, useCallback } from 'react'
import { supabase } from '@/lib/supabase-client'

interface LiveMarketData {
  id: string
  market_type: 'binary' | 'multi_outcome'
  shares_sold: number
  total_shares: number
  yes_pool: number
  no_pool: number
  total_volume: number
  live_volume: number
  yes_price: number
  no_price: number
  outcomes?: Array<{
    id: string
    name: string
    price: number
    pool: number
    volume: number
    color?: string
    icon?: string
  }>
  updated_at: string
}

interface UseLiveMarketUpdatesOptions {
  marketId?: string
  enabled?: boolean
  refreshInterval?: number // in milliseconds
}

export function useLiveMarketUpdates(options: UseLiveMarketUpdatesOptions = {}) {
  const { marketId, enabled = true, refreshInterval = 5000 } = options
  const [marketData, setMarketData] = useState<LiveMarketData | null>(null)
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState<string | null>(null)
  const [lastUpdated, setLastUpdated] = useState<Date | null>(null)

  const fetchMarketData = useCallback(async () => {
    if (!enabled) return

    try {
      setLoading(true)
      setError(null)

      let query = supabase
        .from('markets')
        .select(`
          id,
          market_type,
          shares_sold,
          total_shares,
          yes_pool,
          no_pool,
          total_volume,
          live_volume,
          yes_price,
          no_price,
          updated_at,
          outcomes:market_outcomes(
            id,
            name,
            price,
            pool,
            volume,
            color,
            icon,
            sort_order
          )
        `)

      let result
      if (marketId) {
        result = await query.eq('id', marketId).single()
      } else {
        result = await query.eq('status', 'active')
      }

      const { data, error: fetchError } = result

      if (fetchError) {
        throw fetchError
      }

      if (marketId) {
        // Process single market data
        const marketData = data as any
        if (marketData && marketData.outcomes) {
          // Sort outcomes by sort_order
          marketData.outcomes = marketData.outcomes.sort((a: any, b: any) => 
            (a.sort_order || 0) - (b.sort_order || 0)
          )
        }
        setMarketData(marketData)
      } else {
        // For multiple markets, we'll update the first one or handle as needed
        const marketData = Array.isArray(data) ? data[0] : data as any
        if (marketData && marketData.outcomes) {
          // Sort outcomes by sort_order
          marketData.outcomes = marketData.outcomes.sort((a: any, b: any) => 
            (a.sort_order || 0) - (b.sort_order || 0)
          )
        }
        setMarketData(marketData)
      }

      setLastUpdated(new Date())
    } catch (err: any) {
      console.error('Error fetching live market data:', err)
      setError(err.message)
    } finally {
      setLoading(false)
    }
  }, [marketId, enabled])

  // Initial fetch
  useEffect(() => {
    fetchMarketData()
  }, [fetchMarketData])

  // Set up interval for live updates
  useEffect(() => {
    if (!enabled) return

    const interval = setInterval(() => {
      fetchMarketData()
    }, refreshInterval)

    return () => clearInterval(interval)
  }, [fetchMarketData, enabled, refreshInterval])

  // Set up real-time subscription for immediate updates
  useEffect(() => {
    if (!enabled || !marketId) return

    const channel = supabase
      .channel(`live-market-${marketId}`)
      .on(
        'postgres_changes',
        {
          event: 'UPDATE',
          schema: 'public',
          table: 'markets',
          filter: `id=eq.${marketId}`,
        },
        (payload) => {
          console.log('Live market update received:', payload)
          setMarketData(payload.new as LiveMarketData)
          setLastUpdated(new Date())
        }
      )
      .subscribe()

    return () => {
      supabase.removeChannel(channel)
    }
  }, [marketId, enabled])

  const refreshNow = useCallback(() => {
    fetchMarketData()
  }, [fetchMarketData])

  return {
    marketData,
    loading,
    error,
    lastUpdated,
    refreshNow,
    isLive: lastUpdated && (Date.now() - lastUpdated.getTime()) < refreshInterval + 1000
  }
}

// Hook for multiple markets
export function useLiveMarketsUpdates(options: { enabled?: boolean; refreshInterval?: number } = {}) {
  const { enabled = true, refreshInterval = 5000 } = options
  const [marketsData, setMarketsData] = useState<LiveMarketData[]>([])
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState<string | null>(null)
  const [lastUpdated, setLastUpdated] = useState<Date | null>(null)

  const fetchMarketsData = useCallback(async () => {
    if (!enabled) return

    try {
      setLoading(true)
      setError(null)

      const { data, error: fetchError } = await supabase
        .from('markets')
        .select(`
          id,
          shares_sold,
          total_shares,
          yes_pool,
          no_pool,
          total_volume,
          live_volume,
          yes_price,
          no_price,
          current_yes_share_price,
          current_no_share_price,
          updated_at
        `)
        .eq('status', 'active')
        .order('updated_at', { ascending: false })

      if (fetchError) {
        throw fetchError
      }

      setMarketsData((data as any) || [])
      setLastUpdated(new Date())
    } catch (err: any) {
      console.error('Error fetching live markets data:', err)
      setError(err.message)
    } finally {
      setLoading(false)
    }
  }, [enabled])

  // Initial fetch
  useEffect(() => {
    fetchMarketsData()
  }, [fetchMarketsData])

  // Set up interval for live updates
  useEffect(() => {
    if (!enabled) return

    const interval = setInterval(() => {
      fetchMarketsData()
    }, refreshInterval)

    return () => clearInterval(interval)
  }, [fetchMarketsData, enabled, refreshInterval])

  const refreshNow = useCallback(() => {
    fetchMarketsData()
  }, [fetchMarketsData])

  return {
    marketsData,
    loading,
    error,
    lastUpdated,
    refreshNow,
    isLive: lastUpdated && (Date.now() - lastUpdated.getTime()) < refreshInterval + 1000
  }
}

