import { useState, useEffect, useCallback, useMemo } from 'react'
import { supabase } from '@/lib/supabase-client'

export interface LoginSession {
  id: string
  user_id: string
  session_token: string
  ip_address?: string
  user_agent?: string
  device_type?: string
  browser?: string
  os?: string
  location?: string
  is_active: boolean
  login_time: string
  logout_time?: string
  last_activity: string
  created_at: string
  updated_at: string
}

export function useLoginSessions(userId?: string) {
  const [sessions, setSessions] = useState<LoginSession[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  const fetchSessions = useCallback(async () => {
    console.log('useLoginSessions: fetchSessions called', { userId })
    if (!userId) {
      console.log('useLoginSessions: No userId provided')
      setLoading(false)
      return
    }

    try {
      setLoading(true)
      setError(null)

      console.log('useLoginSessions: Fetching sessions from database...')
      const { data, error: fetchError } = await supabase
        .from('login_sessions')
        .select('*')
        .eq('user_id', userId)
        .order('login_time', { ascending: false })
        .limit(20)

      if (fetchError) {
        console.error('useLoginSessions: Database error:', {
          error: fetchError,
          message: fetchError.message,
          details: fetchError.details,
          hint: fetchError.hint,
          code: fetchError.code
        })
        throw fetchError
      }

      console.log('useLoginSessions: Sessions data received:', data)
      setSessions(data || [])
    } catch (err) {
      console.error('Error fetching login sessions:', err)
      setError(err instanceof Error ? err.message : 'Failed to fetch login sessions')
    } finally {
      setLoading(false)
    }
  }, [userId])

  const createSession = useCallback(async (sessionData: Partial<LoginSession>) => {
    if (!userId) return null

    try {
      const { data, error } = await supabase
        .from('login_sessions')
        .insert({
          user_id: userId,
          ...sessionData
        })
        .select()
        .single()

      if (error) throw error

      // Refresh sessions list
      await fetchSessions()
      return data
    } catch (err) {
      console.error('Error creating login session:', err)
      return null
    }
  }, [userId, fetchSessions])

  const updateSession = useCallback(async (sessionId: string, updates: Partial<LoginSession>) => {
    try {
      const { data, error } = await supabase
        .from('login_sessions')
        .update(updates)
        .eq('id', sessionId)
        .select()
        .single()

      if (error) throw error

      // Update local state
      setSessions(prev => 
        prev.map(session => 
          session.id === sessionId ? { ...session, ...data } : session
        )
      )

      return data
    } catch (err) {
      console.error('Error updating login session:', err)
      return null
    }
  }, [])

  const endSession = useCallback(async (sessionId: string) => {
    return await updateSession(sessionId, {
      is_active: false,
      logout_time: new Date().toISOString()
    })
  }, [updateSession])

  const endAllSessions = useCallback(async () => {
    if (!userId) return false

    try {
      const { error } = await supabase
        .from('login_sessions')
        .update({
          is_active: false,
          logout_time: new Date().toISOString()
        })
        .eq('user_id', userId)
        .eq('is_active', true)

      if (error) throw error

      // Refresh sessions list
      await fetchSessions()
      return true
    } catch (err) {
      console.error('Error ending all sessions:', err)
      return false
    }
  }, [userId, fetchSessions])

  const getCurrentSession = useCallback(() => {
    return sessions.find(session => session.is_active)
  }, [sessions])

  const getActiveSessions = useCallback(() => {
    return sessions.filter(session => session.is_active)
  }, [sessions])

  const getInactiveSessions = useCallback(() => {
    return sessions.filter(session => !session.is_active)
  }, [sessions])

  useEffect(() => {
    fetchSessions()

    // Set up real-time subscription for session updates
    let subscription: any

    if (userId) {
      subscription = supabase
        .channel(`login-sessions-${userId}`)
        .on(
          'postgres_changes',
          {
            event: '*',
            schema: 'public',
            table: 'login_sessions',
            filter: `user_id=eq.${userId}`
          },
          () => {
            fetchSessions()
          }
        )
        .subscribe()
    }

    return () => {
      if (subscription) {
        subscription.unsubscribe()
      }
    }
  }, [userId, fetchSessions])

  // Memoize the return value
  return useMemo(() => ({
    sessions,
    loading,
    error,
    createSession,
    updateSession,
    endSession,
    endAllSessions,
    getCurrentSession,
    getActiveSessions,
    getInactiveSessions,
    refetch: fetchSessions
  }), [
    sessions,
    loading,
    error,
    createSession,
    updateSession,
    endSession,
    endAllSessions,
    getCurrentSession,
    getActiveSessions,
    getInactiveSessions,
    fetchSessions
  ])
}
